const mysql = require('mysql2/promise');
const { promisify } = require('util');
const sleep = promisify(setTimeout);

const MAX_RETRIES = 15;
const INITIAL_RETRY_DELAY = 1000; // 1 second
const MAX_RETRY_DELAY = 60000; // 60 seconds

let pools = new Map();

const configDbPool = mysql.createPool({
  host: 'localhost',
  user: 'sync',
  password: '26iaaHCz6mAswTpW',
  database: 'sync',
  connectionLimit: 10,
  queueLimit: 0,
  waitForConnections: true,
  enableKeepAlive: true,
  keepAliveInitialDelay: 10000,
  connectTimeout: 60000,
});

async function getUserConfig(username) {
  let retries = 0;
  while (retries < MAX_RETRIES) {
    try {
      const [rows] = await configDbPool.execute(
        'SELECT username, password, role, expirationDate, telegramId, dbHost, dbUser, dbPassword, movie_url, series_url, movie_count, series_count FROM user_configs WHERE username = ?',
        [username]
      );
      if (rows.length === 0) {
        throw new Error(`User ${username} not found in database`);
      }
      return rows[0];
    } catch (error) {
      console.error(`Error fetching config for user ${username} (attempt ${retries + 1}):`, error);
      retries++;
      if (retries >= MAX_RETRIES) throw error;
      await sleep(Math.min(INITIAL_RETRY_DELAY * Math.pow(2, retries), MAX_RETRY_DELAY));
    }
  }
}

async function createPool(username) {
  let retries = 0;
  while (retries < MAX_RETRIES) {
    try {
      console.log(`Attempting to create pool for user: ${username} (attempt ${retries + 1})`);
      
      const userConfig = await getUserConfig(username);
      if (!userConfig) {
        throw new Error(`Database configuration missing for user ${username}`);
      }

      if (new Date(userConfig.expirationDate) < new Date()) {
        throw new Error(`User ${username} account has expired`);
      }

      if (!userConfig.dbHost || !userConfig.dbUser || !userConfig.dbPassword) {
        throw new Error(`Incomplete database credentials for user ${username}`);
      }

      const pool = mysql.createPool({
        host: userConfig.dbHost,
        user: userConfig.dbUser,
        password: userConfig.dbPassword,
        database: 'xui',
        connectionLimit: 10,
        queueLimit: 0,
        waitForConnections: true,
        enableKeepAlive: true,
        keepAliveInitialDelay: 10000,
        connectTimeout: 60000,
      });

      // Test the connection
      const connection = await pool.getConnection();
      await connection.ping();
      connection.release();

      console.log(`Successfully created and tested pool for user: ${username}`);
      return pool;
    } catch (error) {
      console.error(`Error creating database pool for user ${username} (attempt ${retries + 1}):`, error);
      retries++;
      if (retries >= MAX_RETRIES) throw error;
      await sleep(Math.min(INITIAL_RETRY_DELAY * Math.pow(2, retries), MAX_RETRY_DELAY));
    }
  }
}

async function getConnection(username) {
  if (!pools.has(username)) {
    pools.set(username, await createPool(username));
  }
  return pools.get(username).getConnection();
}

async function executeQueryWithRetry(connection, query, params, retries = 0) {
  try {
    const [results] = await connection.execute(query, params);
    return results;
  } catch (error) {
    if (retries < MAX_RETRIES && (
      error.code === 'PROTOCOL_CONNECTION_LOST' || 
      error.code === 'ECONNRESET' || 
      error.code === 'PROTOCOL_ENQUEUE_AFTER_FATAL_ERROR' ||
      error.code === 'ECONNREFUSED' ||
      error.message === 'Connection is closed'
    )) {
      console.log(`Retrying query, attempt ${retries + 1} of ${MAX_RETRIES}`);
      await sleep(Math.min(INITIAL_RETRY_DELAY * Math.pow(2, retries), MAX_RETRY_DELAY));
      
      // Get a new connection
      pools.delete(connection.config.user);
      connection = await getConnection(connection.config.user);
      
      return executeQueryWithRetry(connection, query, params, retries + 1);
    }
    throw error;
  }
}

async function executeQuery(username, query, params = []) {
  let connection;
  try {
    connection = await getConnection(username);
    return await executeQueryWithRetry(connection, query, params);
  } catch (error) {
    console.error('Database query error:', error);
    throw error;
  } finally {
    if (connection) {
      try {
        connection.release();
      } catch (releaseError) {
        console.error('Error releasing connection:', releaseError);
      }
    }
  }
}

async function executeTransaction(username, callback) {
  let connection;
  let retries = 0;
  while (retries < MAX_RETRIES) {
    try {
      connection = await getConnection(username);
      await connection.beginTransaction();
      
      const result = await callback(connection);
      
      await connection.commit();
      return result;
    } catch (error) {
      console.error(`Transaction error (attempt ${retries + 1}):`, error);
      if (connection) {
        try {
          await connection.rollback();
        } catch (rollbackError) {
          console.error('Error rolling back transaction:', rollbackError);
        }
      }
      
      retries++;
      if (retries >= MAX_RETRIES) throw error;
      await sleep(Math.min(INITIAL_RETRY_DELAY * Math.pow(2, retries), MAX_RETRY_DELAY));
      
      // Recreate the pool for this user
      pools.delete(username);
    } finally {
      if (connection) {
        try {
          connection.release();
        } catch (releaseError) {
          console.error('Error releasing connection:', releaseError);
        }
      }
    }
  }
}

async function closeAllPools() {
  for (const [username, pool] of pools.entries()) {
    try {
      await pool.end();
      console.log(`Closed pool for user: ${username}`);
    } catch (error) {
      console.error(`Error closing pool for user ${username}:`, error);
    }
  }
  pools.clear();
  
  try {
    await configDbPool.end();
    console.log('Closed config database pool');
  } catch (error) {
    console.error('Error closing config database pool:', error);
  }
}

process.on('SIGINT', async () => {
  console.log('Closing all database connections...');
  await closeAllPools();
  process.exit(0);
});

module.exports = {
  executeQuery,
  executeTransaction,
  closeAllPools,
  getUserConfig,
  configDbPool
};