const socketIo = require('socket.io');
const jwt = require('jsonwebtoken');

let io;
const userProcesses = new Map();

const initializeSocket = (server) => {
  io = socketIo(server, {
    cors: {
      origin: "http://sync.zpainel.site:3000",
      methods: ["GET", "POST"],
      credentials: true
    },
    pingTimeout: 60000,
    pingInterval: 25000,
    transports: ['websocket', 'polling'],
    allowEIO3: true
  });

  io.use((socket, next) => {
    if (socket.handshake.query && socket.handshake.query.token) {
      jwt.verify(socket.handshake.query.token, process.env.JWT_SECRET, (err, decoded) => {
        if (err) {
          console.error('JWT verification failed:', err);
          return next(new Error('Authentication error'));
        }
        socket.decoded = decoded;
        next();
      });
    } else {
      console.error('No token provided in handshake');
      next(new Error('Authentication error'));
    }
  });

  io.on('connection', (socket) => {
    console.log(`New client connected: ${socket.decoded.username}`);
    socket.join(socket.decoded.username);

    // Enviar o estado atual do processo, se existir
    const currentProcess = userProcesses.get(socket.decoded.username);
    if (currentProcess) {
      socket.emit('processUpdate', currentProcess);
    }

    socket.on('disconnect', (reason) => {
      console.log(`Client disconnected: ${socket.decoded.username}, reason: ${reason}`);
    });

    socket.on('error', (error) => {
      console.error(`Socket error for ${socket.decoded.username}:`, error);
    });

    // Adicionar um evento de reconexão
    socket.on('reconnect', (attemptNumber) => {
      console.log(`Client reconnected: ${socket.decoded.username}, attempt: ${attemptNumber}`);
    });

    // Adicionar um evento de erro de reconexão
    socket.on('reconnect_error', (error) => {
      console.error(`Reconnection error for ${socket.decoded.username}:`, error);
    });
  });

  return io;
};

const updateUserProcess = (username, processData) => {
  userProcesses.set(username, processData);
  if (io) {
    io.to(username).emit('processUpdate', processData);
  } else {
    console.error('Socket.io not initialized when trying to update user process');
  }
};

const removeUserProcess = (username) => {
  userProcesses.delete(username);
};

const getUserProcess = (username) => {
  return userProcesses.get(username);
};

const getIo = () => {
  if (!io) {
    console.error('Socket.io not initialized when getIo called');
  }
  return io;
};

module.exports = {
  initializeSocket,
  getIo,
  updateUserProcess,
  removeUserProcess,
  getUserProcess
};