const axios = require('axios');
const stringSimilarity = require('string-similarity');
const tmdbKeyManager = require('./tmdbKeyManager');

async function getTMDbInfo(title, year = null) {
    let allResults = [];
    const languages = ['pt-BR', 'en-US'];

    for (const language of languages) {
        for (const useYear of [false, true]) {
            try {
                const searchTitle = useYear && year ? `${title} ${year}` : title;
                const results = await searchTMDb(searchTitle, useYear ? year : null, language);
                if (results) {
                    allResults.push(...results);
                }
            } catch (error) {
                console.error(`Error searching TMDB for ${language} with${useYear ? '' : 'out'} year:`, error.message);
            }
        }
    }

    if (allResults.length === 0) {
        return null;
    }

    const bestMatch = findBestMatch(title, allResults);
    return bestMatch ? await getSeriesDetails(bestMatch.id) : null;
}

async function makeApiRequest(endpoint, params) {
    const apiKey = await tmdbKeyManager.getNextAvailableKey();
    try {
        const response = await axios.get(`https://api.themoviedb.org/3/${endpoint}`, {
            params: { ...params, api_key: apiKey }
        });
        return response.data;
    } catch (error) {
        console.error(`Error with TMDB API:`, error.message);
        throw error;
    }
}

async function searchTMDb(title, year, language) {
    try {
        const data = await makeApiRequest('search/tv', {
            query: title,
            first_air_date_year: year,
            language: language
        });
        return data.results;
    } catch (error) {
        console.error('Error in searchTMDb:', error.message);
        return null;
    }
}

async function getSeriesDetails(seriesId) {
    try {
        const data = await makeApiRequest(`tv/${seriesId}`, {
            append_to_response: 'credits,external_ids',
            language: 'pt-BR'
        });

        return {
            name: data.name || '',
            original_name: data.original_name || '',
            overview: data.overview || '',
            poster_path: data.poster_path || '',
            backdrop_path: data.backdrop_path || '',
            first_air_date: data.first_air_date || '',
            vote_average: data.vote_average || 0,
            genres: data.genres.map(genre => genre.name) || [],
            cast: data.credits.cast.slice(0, 5).map(actor => actor.name) || [],
            created_by: data.created_by.map(creator => creator.name) || [],
            number_of_seasons: data.number_of_seasons || 0,
            number_of_episodes: data.number_of_episodes || 0,
            episode_run_time: data.episode_run_time ? data.episode_run_time[0] : null,
            in_production: data.in_production || false,
            status: data.status || '',
            imdb_id: data.external_ids.imdb_id || '',
            id: data.id
        };
    } catch (error) {
        console.error(`Error getting series details:`, error.message);
        return null;
    }
}

function findBestMatch(title, results) {
    const similarities = results.map(result => ({
        result,
        similarity: Math.max(
            stringSimilarity.compareTwoStrings(title.toLowerCase(), result.name.toLowerCase()),
            stringSimilarity.compareTwoStrings(title.toLowerCase(), result.original_name.toLowerCase())
        )
    }));

    similarities.sort((a, b) => b.similarity - a.similarity);
    return similarities[0]?.result;
}

async function getTMDbEpisodeInfo(seriesId, seasonNum, episodeNum) {
    try {
        const data = await makeApiRequest(`tv/${seriesId}/season/${seasonNum}/episode/${episodeNum}`, {
            language: 'pt-BR'
        });

        return {
            name: data.name,
            overview: data.overview,
            air_date: data.air_date,
            still_path: data.still_path,
            vote_average: data.vote_average
        };
    } catch (error) {
        console.error(`Error getting episode details: ${error.message}`);
        return null;
    }
}

module.exports = {
    getTMDbInfo,
    getTMDbEpisodeInfo
};