const axios = require('axios');
const fs = require('fs').promises;
const path = require('path');
const stringSimilarity = require('string-similarity');
const tmdbKeyManager = require('./tmdbKeyManager');

const CACHE_DIR = path.join(__dirname, 'tmdb-filmes');
const CACHE_DURATION = 7 * 24 * 60 * 60 * 1000; // 7 dias em milissegundos

async function ensureCacheDir() {
    try {
        await fs.mkdir(CACHE_DIR, { recursive: true });
    } catch (error) {
        console.error('Erro ao criar diretório de cache:', error);
    }
}

async function getCachedMovie(movieId) {
    const filePath = path.join(CACHE_DIR, `${movieId}.json`);
    try {
        const data = await fs.readFile(filePath, 'utf8');
        const movie = JSON.parse(data);
        const now = new Date();
        const cacheDate = new Date(movie.cacheDate);
        if (now - cacheDate < CACHE_DURATION) {
            console.log(`Usando dados em cache para o filme ${movieId}`);
            return { movie, fromCache: true };
        } else {
            console.log(`Cache expirado para o filme ${movieId}, buscando novos dados`);
        }
    } catch (error) {
        if (error.code !== 'ENOENT') {
            console.error(`Erro ao ler cache para o filme ${movieId}:`, error);
        }
    }
    return { movie: null, fromCache: false };
}

async function cacheMovie(movieId, movieData) {
    const filePath = path.join(CACHE_DIR, `${movieId}.json`);
    const dataToCache = {
        ...movieData,
        cacheDate: new Date().toISOString()
    };
    try {
        await fs.writeFile(filePath, JSON.stringify(dataToCache, null, 2));
        console.log(`Dados em cache salvos para o filme ${movieId}`);
    } catch (error) {
        console.error(`Erro ao cachear filme ${movieId}:`, error);
    }
}

async function getTMDbInfo(title, year = null) {
    await ensureCacheDir();
    let allResults = [];
    const languages = ['pt-BR', 'en-US'];
    
    await Promise.all(languages.map(async (language) => {
        try {
            const [resultWithYear, resultWithoutYear] = await Promise.all([
                year ? searchTMDb(title, year, language) : Promise.resolve(null),
                searchTMDb(title, null, language)
            ]);
            
            allResults = [...allResults, ...(resultWithYear || []), ...(resultWithoutYear || [])];
        } catch (error) {
            console.error(`Erro ao buscar no TMDb para ${language}:`, error.message);
        }
    }));

    if (allResults.length === 0) {
        console.log(`Nenhum resultado encontrado para "${title}" ${year ? `(${year})` : ''}`);
        return { tmdbInfo: null, fromCache: false };
    }

    const bestMatch = findBestMatch(title, allResults);
    if (!bestMatch) {
        console.log(`Nenhuma correspondência encontrada para "${title}" ${year ? `(${year})` : ''}`);
        return { tmdbInfo: null, fromCache: false };
    }

    const { movie: cachedMovie, fromCache } = await getCachedMovie(bestMatch.id);
    if (cachedMovie) {
        return { tmdbInfo: cachedMovie, fromCache };
    }

    console.log(`Buscando novos dados para o filme ${bestMatch.id}`);
    const movieDetails = await getMovieDetails(bestMatch.id);
    if (movieDetails) {
        await cacheMovie(bestMatch.id, movieDetails);
    }
    return { tmdbInfo: movieDetails, fromCache: false };
}

async function makeApiRequest(endpoint, params) {
    const apiKey = await tmdbKeyManager.getAvailableKey();
    try {
        const response = await axios.get(`https://api.themoviedb.org/3/${endpoint}`, {
            params: { ...params, api_key: apiKey }
        });
        return response.data;
    } catch (error) {
        console.error(`Erro na API do TMDb:`, error.message);
        throw error;
    }
}

async function searchTMDb(title, year, language) {
    try {
        const data = await makeApiRequest('search/movie', {
            query: title,
            year: year,
            language: language
        });
        return data.results;
    } catch (error) {
        console.error('Erro ao pesquisar no TMDb:', error.message);
        return null;
    }
}

async function getMovieDetails(movieId) {
    try {
        const data = await makeApiRequest(`movie/${movieId}`, {
            append_to_response: 'credits',
            language: 'pt-BR'
        });
        return {
            id: data.id,
            title: data.title,
            original_title: data.original_title,
            overview: data.overview,
            poster_path: data.poster_path,
            backdrop_path: data.backdrop_path,
            release_date: data.release_date,
            vote_average: data.vote_average,
            genres: data.genres.map(genre => genre.name),
            cast: data.credits.cast.slice(0, 5).map(actor => actor.name),
            director: data.credits.crew.find(person => person.job === 'Director')?.name
        };
    } catch (error) {
        console.error('Erro ao obter detalhes do filme:', error.message);
        return null;
    }
}

function findBestMatch(title, results) {
    const similarities = results.map(result => ({
        result,
        similarity: Math.max(
            stringSimilarity.compareTwoStrings(title.toLowerCase(), result.title.toLowerCase()),
            stringSimilarity.compareTwoStrings(title.toLowerCase(), result.original_title.toLowerCase())
        )
    }));
    similarities.sort((a, b) => b.similarity - a.similarity);
    return similarities[0]?.result;
}

module.exports = {
    getTMDbInfo,
    clearCache: async () => {
        try {
            const files = await fs.readdir(CACHE_DIR);
            await Promise.all(files.map(file => fs.unlink(path.join(CACHE_DIR, file))));
            console.log('Cache limpo com sucesso');
        } catch (error) {
            console.error('Erro ao limpar o cache:', error);
        }
    }
};