const axios = require('axios');
const { executeQuery, executeTransaction } = require('../utils/database');
const { getIo } = require('../socket');
const { getTMDbInfo, getTMDbEpisodeInfo } = require('../services/tmdb-series');

const userProcessingStates = {};


exports.getCategories = async (req, res) => {
    const { url } = req.body;
    try {
        const { username, password, server } = parseXtreamUrl(url);
        console.log(`Obtendo categorias de: ${server}`);

        const categoriesResponse = await axios.get(`${server}/player_api.php`, {
            params: { username, password, action: 'get_series_categories' },
            timeout: 10000
        });

        const seriesResponse = await axios.get(`${server}/player_api.php`, {
            params: { username, password, action: 'get_series' },
            timeout: 10000
        });

        const categories = categoriesResponse.data || [];
        const series = seriesResponse.data || {};

        // Contar séries por categoria
        const seriesCountByCategory = {};
        Object.values(series).forEach(serie => {
            if (serie.category_id) {
                seriesCountByCategory[serie.category_id] = (seriesCountByCategory[serie.category_id] || 0) + 1;
            }
        });

        // Adicionar contagem de séries a cada categoria
        const categoriesWithCount = categories.map(category => ({
            ...category,
            series_count: seriesCountByCategory[category.category_id] || 0
        }));

        const totalSeries = Object.keys(series).length;

        console.log(`Categorias obtidas: ${categoriesWithCount.length}, Total de séries: ${totalSeries}`);

        res.json({ categories: categoriesWithCount, totalSeries });
    } catch (error) {
        console.error('Erro ao obter categorias:', error);
        res.status(500).json({ message: 'Erro ao obter categorias', error: error.message });
    }
};

exports.getBouquets = async (req, res) => {
    const { username } = req.user;
    try {
        const bouquets = await executeQuery(username, 'SELECT id, bouquet_name FROM bouquets');
        if (!bouquets || bouquets.length === 0) {
            return res.status(404).json({ message: 'No bouquets found' });
        }
        res.json(bouquets);
    } catch (error) {
        res.status(500).json({ message: 'Error fetching bouquets', error: error.message });
    }
};

exports.processSeries = async (req, res) => {
    const { url, bouquetId, selectedCategories } = req.body;
    const { username } = req.user;
    const processingState = getProcessingState(username);

    if (processingState.isRunning) {
        return res.status(400).json({ message: 'Já existe um processamento em andamento' });
    }

    if (!selectedCategories || selectedCategories.length === 0) {
        return res.status(400).json({ message: 'Nenhuma categoria selecionada' });
    }

    try {
        const { username: xtreamUser, password: xtreamPass, server } = parseXtreamUrl(url);
        
        const series = await getSeries(xtreamUser, xtreamPass, server);
        const totalItems = series.filter(seriesItem => 
            selectedCategories.some(sc => sc.id === seriesItem.category_id && sc.selected)
        ).length;

        initializeProcessingState(username, totalItems);
        res.json({ message: 'Processamento iniciado', totalItems: totalItems });

        console.log(`Iniciando processamento de séries para ${username}...`);

        const categoryMapping = await processCategories(xtreamUser, xtreamPass, server, username, selectedCategories);

        for (const seriesItem of series) {
            if (await shouldContinueProcessing(username) === false) break;

            const selectedCategory = selectedCategories.find(sc => sc.id === seriesItem.category_id && sc.selected);
            if (!selectedCategory) continue;

            console.log(`Processando série: ${seriesItem.name} (ID: ${seriesItem.series_id})`);

            try {
                await executeTransaction(username, async (connection) => {
                    const result = await processSeriesItem(connection, seriesItem, categoryMapping, xtreamUser, xtreamPass, server, username, selectedCategory);
                    if (result.seriesId) {
                        await updateBouquet(connection, result.seriesId, bouquetId);
                    }
                    logSeriesResult(username, result);
                });

                updateProcessingState(username, seriesItem.name);
                emitProgressUpdate(username);
            } catch (error) {
                console.error(`Erro ao processar série ${seriesItem.name}:`, error);
            }
        }

        finishProcessing(username);
    } catch (error) {
        handleProcessingError(error, username);
    }
};
function logSeriesResult(username, result) {
    const processingState = getProcessingState(username);
    if (result.added) {
        const logMessage = `Nova série adicionada: ${result.seriesName} - ${result.seasonCount} Temporadas, ${result.episodeCount} Episódios`;
        processingState.log.push(logMessage);
        console.log(logMessage);
    } else if (result.newEpisodes > 0) {
        const logMessage = `Série ${result.seriesName}: ${result.newEpisodes} novos episódios adicionados`;
        processingState.log.push(logMessage);
        console.log(logMessage);
    }
}

async function processCategories(xtreamUser, xtreamPass, server, username, selectedCategories) {
    console.log('Obtendo categorias...');
    try {
        const categoryMapping = {};
        for (const selectedCategory of selectedCategories) {
            if (!selectedCategory.selected) continue;

            try {
                console.log(`Processando categoria: ${selectedCategory.category_name}`);
                
                // Verifica se a categoria já existe no banco de dados
                const existingCategoryResult = await executeQuery(username, 
                    'SELECT id FROM streams_categories WHERE category_type = ? AND category_name = ?',
                    ['series', selectedCategory.newName || selectedCategory.category_name]
                );

                let categoryId;
                if (Array.isArray(existingCategoryResult) && existingCategoryResult.length > 0) {
                    // Se a categoria já existe, usa o ID existente
                    categoryId = existingCategoryResult[0].id;
                    console.log(`Categoria existente encontrada: ${selectedCategory.newName || selectedCategory.category_name}, ID: ${categoryId}`);
                } else {
                    // Se a categoria não existe, cria uma nova
                    const insertResult = await executeQuery(username, 
                        'INSERT INTO streams_categories (category_type, category_name, parent_id, cat_order, is_adult) VALUES (?, ?, ?, ?, ?)',
                        ['series', selectedCategory.newName || selectedCategory.category_name, 0, 0, 0]
                    );
                    categoryId = insertResult.insertId;
                    console.log(`Nova categoria criada: ${selectedCategory.newName || selectedCategory.category_name}, ID: ${categoryId}`);
                }

                categoryMapping[selectedCategory.id] = categoryId;
                console.log(`Categoria mapeada: API ID ${selectedCategory.id} -> DB ID ${categoryId}`);
            } catch (error) {
                console.error(`Erro ao processar categoria ${selectedCategory.category_name}:`, error);
                console.error('Stack trace:', error.stack);
            }
        }
        console.log(`Categorias processadas: ${Object.keys(categoryMapping).length}`);
        return categoryMapping;
    } catch (error) {
        console.error('Erro ao obter ou processar categorias:', error);
        console.error('Stack trace:', error.stack);
        return {};
    }
}

async function getSeries(xtreamUser, xtreamPass, server) {
    console.log('Obtaining series list...');
    
    const fetchSeries = async () => {
        const response = await axios.get(`${server}/player_api.php`, {
            params: { 
                username: xtreamUser, 
                password: xtreamPass, 
                action: 'get_series' 
            },
            timeout: 15000
        });

        if (!response.data || typeof response.data !== 'object') {
            throw new Error('Invalid series data format received');
        }

        const series = Object.values(response.data);
        console.log(`Total series obtained: ${series.length}`);
        return series;
    };

    return await withRetry(fetchSeries);
}

async function processSeriesItem(connection, seriesItem, categoryMapping, xtreamUser, xtreamPass, server, username, selectedCategory) {
    const mappedCategoryId = categoryMapping[seriesItem.category_id];
    if (!mappedCategoryId) {
        console.error(`Categoria não mapeada para a série: ${seriesItem.name}`);
        return { added: false, newEpisodes: 0 };
    }

    const yearMatch = seriesItem.name.match(/\((\d{4})\)$/);
    const year = yearMatch ? parseInt(yearMatch[1]) : null;
    const cleanedSeriesName = seriesItem.name.replace(/\s*\(\d{4}\)$/, '').trim();

    const [existingSeries] = await connection.query(
        'SELECT id, tmdb_id FROM streams_series WHERE title = ? AND category_id = ?',
        [cleanedSeriesName, `[${mappedCategoryId}]`]
    );

    let seriesId, tmdbId;
    let isNewSeries = false;

    if (existingSeries.length > 0) {
        console.log(`Série "${cleanedSeriesName}" já existe. Verificando novos episódios.`);
        seriesId = existingSeries[0].id;
        tmdbId = existingSeries[0].tmdb_id;
    } else {
        const tmdbData = await getTMDbInfo(cleanedSeriesName, year);

        const seriesInfo = {
            title: cleanedSeriesName,
            year: tmdbData?.first_air_date ? new Date(tmdbData.first_air_date).getFullYear() : year,
            cover: tmdbData?.poster_path ? `https://image.tmdb.org/t/p/w300${tmdbData.poster_path}` : (seriesItem.cover || ''),
            cover_big: tmdbData?.backdrop_path ? `https://image.tmdb.org/t/p/w780${tmdbData.backdrop_path}` : (seriesItem.cover || ''),
            genre: tmdbData?.genres?.join(', ') || '',
            plot: tmdbData?.overview || seriesItem.plot || '',
            cast: tmdbData?.cast?.join(', ') || '',
            rating: tmdbData?.vote_average || seriesItem.rating || 0,
            director: tmdbData?.created_by?.join(', ') || '',
            tmdb_id: tmdbData?.id || null,
            release_date: tmdbData?.first_air_date || null,
            episode_run_time: tmdbData?.episode_run_time || 0,
            backdrop_path: tmdbData?.backdrop_path ? `https://image.tmdb.org/t/p/original${tmdbData.backdrop_path}` : null,
            youtube_trailer: null
        };

        const [result] = await connection.query(
            `INSERT INTO streams_series 
            (title, category_id, cover, cover_big, genre, plot, cast, rating, director, release_date, tmdb_id, episode_run_time, backdrop_path, youtube_trailer, year)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                seriesInfo.title,
                `[${mappedCategoryId}]`,
                seriesInfo.cover,
                seriesInfo.cover_big,
                seriesInfo.genre,
                seriesInfo.plot,
                seriesInfo.cast,
                seriesInfo.rating,
                seriesInfo.director,
                seriesInfo.release_date,
                seriesInfo.tmdb_id,
                seriesInfo.episode_run_time,
                seriesInfo.backdrop_path,
                seriesInfo.youtube_trailer,
                seriesInfo.year
            ]
        );

        seriesId = result.insertId;
        tmdbId = seriesInfo.tmdb_id;
        console.log(`Nova série adicionada: ${seriesInfo.title}, ID: ${seriesId}, TMDB ID: ${tmdbId || 'N/A'}`);
        
        isNewSeries = true;
    }

    const episodeStats = await processEpisodes(connection, seriesId, seriesItem, xtreamUser, xtreamPass, server, mappedCategoryId, cleanedSeriesName, tmdbId, username);

    return {
        seriesId,
        added: isNewSeries,
        newEpisodes: episodeStats.addedEpisodes,
        seasonCount: episodeStats.seasonCount,
        episodeCount: episodeStats.episodeCount
    };
}

async function updateExistingSeries(connection, existingSeries, seriesItem, xtreamUser, xtreamPass, server, username) {
    const episodeStats = await processEpisodes(connection, existingSeries.id, seriesItem, xtreamUser, xtreamPass, server, existingSeries.tmdb_id, username);
    console.log(`Série atualizada: ${seriesItem.name} - Novos episódios: ${episodeStats.addedEpisodes}`);
    return existingSeries.id;
}
async function processEpisodes(connection, seriesId, seriesItem, xtreamUser, xtreamPass, server, mappedCategoryId, cleanedSeriesName, tmdbId, username) {
    console.log(`Getting episode information for series: ${seriesItem.name}`);
    
    const fetchEpisodes = async () => {
        const response = await axios.get(`${server}/player_api.php`, {
            params: {
                username: xtreamUser,
                password: xtreamPass,
                action: 'get_series_info',
                series_id: seriesItem.series_id
            },
            timeout: 15000
        });

        if (!response.data || !response.data.episodes) {
            throw new Error('Invalid episode data format received');
        }

        return response.data;
    };

    const episodesData = await withRetry(fetchEpisodes);
    let seasonCount = 0;
    let episodeCount = 0;
    let addedEpisodes = 0;

    if (episodesData.episodes) {
        const processingState = getProcessingState(username);
        const totalEpisodes = Object.values(episodesData.episodes).flat().length;
        processingState.totalEpisodes += totalEpisodes;

        for (const seasonNum in episodesData.episodes) {
            if (!Object.prototype.hasOwnProperty.call(episodesData.episodes, seasonNum)) continue;
            
            seasonCount++;
            const season = episodesData.episodes[seasonNum];
            
            if (!Array.isArray(season)) {
                console.warn(`Invalid format for season ${seasonNum}`);
                continue;
            }

            for (const episode of season) {
                try {
                    const result = await withRetry(() => 
                        processEpisode(
                            connection, 
                            seriesId, 
                            seasonNum, 
                            episode, 
                            server, 
                            xtreamUser, 
                            xtreamPass, 
                            cleanedSeriesName, 
                            tmdbId, 
                            username, 
                            mappedCategoryId
                        )
                    );
                    
                    if (result.added) addedEpisodes++;
                    episodeCount++;
                    processingState.processedEpisodes++;
                } catch (error) {
                    console.error(`Error processing episode S${seasonNum}E${episode.episode_num}:`, error);
                    // Continue with next episode instead of failing entire series
                }
            }
        }
    }

    return { seasonCount, episodeCount, addedEpisodes };
}

// Improved processEpisode function with enhanced error handling
async function processEpisode(connection, seriesId, seasonNum, episode, server, xtreamUser, xtreamPass, seriesName, tmdbId, username, mappedCategoryId) {
    console.log(`Processing episode: S${seasonNum}E${episode.episode_num}`);
    
    try {
        const [existingEpisode] = await connection.query(
            'SELECT id FROM streams_episodes WHERE series_id = ? AND season_num = ? AND episode_num = ?',
            [seriesId, seasonNum, episode.episode_num]
        );

        if (existingEpisode.length > 0) {
            console.log(`Episode S${seasonNum}E${episode.episode_num} already exists for series ${seriesName}.`);
            return { added: false };
        }

        let episodeIcon = episode.info?.movie_image || '';
        let episodeOverview = '';
        let episodeName = '';

        if (tmdbId) {
            try {
                const tmdbEpisodeInfo = await withRetry(() => 
                    getTMDbEpisodeInfo(tmdbId, seasonNum, episode.episode_num)
                );
                
                if (tmdbEpisodeInfo) {
                    episodeIcon = tmdbEpisodeInfo.still_path ? 
                        `https://image.tmdb.org/t/p/w300${tmdbEpisodeInfo.still_path}` : episodeIcon;
                    episodeOverview = tmdbEpisodeInfo.overview || '';
                    episodeName = tmdbEpisodeInfo.name || '';
                }
            } catch (error) {
                console.error(`Error fetching TMDB episode info: ${error.message}`);
                // Continue with default values if TMDB fails
            }
        }

        episodeName = episodeName ? 
            `S${seasonNum}E${episode.episode_num} - ${episodeName}` :
            `S${seasonNum}E${episode.episode_num} - ${seriesName}`;

        const movieProperties = JSON.stringify({
            release_date: "",
            plot: episodeOverview,
            duration_secs: episode.info?.duration_secs || 0,
            duration: episode.info?.duration || "00:00:00",
            movie_image: episodeIcon,
            video: [],
            audio: [],
            bitrate: 0,
            rating: "",
            season: seasonNum,
            tmdb_id: tmdbId || ""
        });

        // Use transaction for atomic episode insertion
        await connection.beginTransaction();

        const [streamResult] = await connection.query(
            `INSERT INTO streams 
            (type, category_id, stream_display_name, stream_source, stream_icon, notes, 
            enable_transcode, movie_properties, read_native, target_container, stream_all, 
            remove_subtitles, direct_source, added, series_no, tmdb_language, year, rating)
            VALUES (5, ?, ?, ?, ?, ?, 0, ?, 0, 'mp4', 0, 0, 1, ?, ?, 'pt-br', NULL, 0)`,
            [
                `[${mappedCategoryId}]`,
                episodeName,
                JSON.stringify([`${server}/series/${xtreamUser}/${xtreamPass}/${episode.id}.${episode.container_extension}`]),
                episodeIcon,
                episodeOverview,
                movieProperties,
                Math.floor(Date.now() / 1000),
                seriesId
            ]
        );

        const streamId = streamResult.insertId;

        await connection.query(
            `INSERT INTO streams_episodes (stream_id, series_id, season_num, episode_num)
            VALUES (?, ?, ?, ?)`,
            [streamId, seriesId, seasonNum, episode.episode_num]
        );

        await connection.commit();

        console.log(`New episode added: ${episodeName}, Stream ID: ${streamId}`);
        const processingState = getProcessingState(username);
        processingState.addedEpisodes++;
        
        return { added: true };
    } catch (error) {
        if (connection) {
            await connection.rollback();
        }
        throw error; // Re-throw to be handled by the retry mechanism
    }
}



async function updateBouquet(connection, seriesId, bouquetId) {
    await connection.query(
        'UPDATE bouquets SET bouquet_series = JSON_ARRAY_APPEND(COALESCE(bouquet_series, "[]"), "$", ?) WHERE id = ?',
        [seriesId, bouquetId]
    );
}

exports.pauseProcess = (req, res) => {
    const { username } = req.user;
    const processingState = getProcessingState(username);
    if (!processingState.isRunning) {
        return res.status(400).json({ message: 'Nenhum processamento em andamento' });
    }
    processingState.isPaused = true;
    res.json({ message: 'Processamento pausado' });
};

exports.resumeProcess = (req, res) => {
    const { username } = req.user;
    const processingState = getProcessingState(username);
    if (!processingState.isRunning) {
        return res.status(400).json({ message: 'Nenhum processamento em andamento' });
    }
    processingState.isPaused = false;
    res.json({ message: 'Processamento retomado' });
};

exports.cancelProcess = (req, res) => {
    const { username } = req.user;
    const processingState = getProcessingState(username);
    
    // Sempre permita cancelar, independente do estado
    processingState.isRunning = false;
    processingState.isPaused = false;
    processingState.isCompleted = false;
    
    const finalResults = {
        totalProcessed: processingState.processedItems || 0,
        totalAdded: processingState.addedItems || 0,
        totalDuplicates: processingState.duplicateSeries || 0,
        totalEpisodes: processingState.totalEpisodes || 0,
        processedEpisodes: processingState.processedEpisodes || 0,
        addedEpisodes: processingState.addedEpisodes || 0,
        duplicateEpisodes: processingState.duplicateEpisodes || 0
    };

    res.json({ 
        message: 'Processamento finalizado',
        finalResults: finalResults
    });
    
    // Remove the user's processing state immediately
    delete userProcessingStates[username];
    getIo().to(username).emit('processCancelled', {
        message: 'Processamento cancelado e dados limpos.'
    });
};

exports.getProcessStatus = (req, res) => {
    const { username } = req.user;
    const processingState = getProcessingState(username);
    if (processingState.isRunning) {
        res.json({
            isRunning: true,
            isPaused: processingState.isPaused,
            progress: {
                processedItems: processingState.processedItems,
                addedItems: processingState.addedItems,
                timeRemaining: estimateTimeRemaining(username),
                completedSeries: processingState.completedSeries.slice(-5),
                totalItems: processingState.totalItems,
                totalEpisodes: processingState.totalEpisodes,
                processedEpisodes: processingState.processedEpisodes
            }
        });
    } else {
        res.json({ isRunning: false });
    }
};

function parseXtreamUrl(url) {
    const parsedUrl = new URL(url);
    return {
        username: parsedUrl.searchParams.get('username'),
        password: parsedUrl.searchParams.get('password'),
        server: `${parsedUrl.protocol}//${parsedUrl.hostname}:${parsedUrl.port}`
    };
}

function estimateTimeRemaining(username) {
    const processingState = getProcessingState(username);
    const elapsedTime = (Date.now() - processingState.startTime) / 1000; // em segundos
    const processedPerSecond = processingState.processedItems / elapsedTime;
    
    if (processedPerSecond === 0 || isNaN(processedPerSecond) || processingState.totalItems === 0) {
        return 'Calculando...';
    }

    const remainingItems = processingState.totalItems - processingState.processedItems;
    const remainingSeconds = Math.round(remainingItems / processedPerSecond);

    const hours = Math.floor(remainingSeconds / 3600);
    const minutes = Math.floor((remainingSeconds % 3600) / 60);

    if (hours > 0) {
        return `${hours}h ${minutes}m`;
    } else if (minutes > 0) {
        return `${minutes}m`;
    } else {
        return 'Menos de 1 minuto';
    }
}

function initializeProcessingState(username, totalItems) {
    userProcessingStates[username] = {
        isRunning: true,
        isPaused: false,
        processedItems: 0,
        addedItems: 0,
        duplicateSeries: 0,
        totalItems: totalItems,
        completedSeries: [],
        startTime: Date.now(),
        totalEpisodes: 0,
        processedEpisodes: 0,
        addedEpisodes: 0,
        duplicateEpisodes: 0
    };
}

async function shouldContinueProcessing(username) {
    const processingState = getProcessingState(username);
    if (!processingState.isRunning) {
        console.log(`Processamento interrompido pelo usuário ${username}.`);
        return false;
    }
    while (processingState.isPaused) {
        await new Promise(resolve => setTimeout(resolve, 1000));
        if (!processingState.isRunning) return false;
    }
    return true;
}

function updateProcessingState(username, seriesInfo) {
    const processingState = getProcessingState(username);
    processingState.processedItems++;

    if (seriesInfo) {
        // Extrai apenas o nome da série, removendo informações de temporadas e episódios
        const seriesName = seriesInfo.split(' - ')[0].trim();

        // Verifica se a série já está na lista de completadas
        if (!processingState.completedSeries.includes(seriesName)) {
            processingState.completedSeries.push(seriesName);
        }
    }

    emitProgressUpdate(username);
}

// Função auxiliar para obter o estado de processamento
function getProcessingState(username) {
    if (!userProcessingStates[username]) {
        userProcessingStates[username] = {
            isRunning: false,
            isPaused: false,
            isCompleted: false,
            processedItems: 0,
            addedItems: 0,
            duplicateSeries: 0,
            totalItems: 0,
            completedSeries: [],
            startTime: null,
            totalEpisodes: 0,
            processedEpisodes: 0,
            addedEpisodes: 0,
            duplicateEpisodes: 0,
            timeRemaining: 0,
            newSeries: [],
            log: []
        };
    }
    return userProcessingStates[username];
}

// Função para emitir atualizações de progresso
function emitProgressUpdate(username) {
    const processingState = getProcessingState(username);
    getIo().to(username).emit('processUpdate', {
        timeRemaining: estimateTimeRemaining(username),
        processedItems: processingState.processedItems,
        addedItems: processingState.addedItems,
        duplicateSeries: processingState.duplicateSeries,
        completedSeries: processingState.completedSeries.slice(-5),
        totalItems: processingState.totalItems,
        totalEpisodes: processingState.totalEpisodes,
        processedEpisodes: processingState.processedEpisodes,
        addedEpisodes: processingState.addedEpisodes,
        duplicateEpisodes: processingState.duplicateEpisodes
    });
}

function finishProcessing(username) {
    const processingState = getProcessingState(username);
    processingState.isRunning = false;
    console.log(`Processamento de séries concluído para ${username}`);
    console.log(`Total de séries processadas: ${processingState.processedItems}`);
    console.log(`Total de episódios processados: ${processingState.processedEpisodes}`);
    getIo().to(username).emit('processComplete', {
        totalSeries: processingState.processedItems,
        totalEpisodes: processingState.totalEpisodes,
        processedEpisodes: processingState.processedEpisodes,
        completedSeries: processingState.completedSeries
    });
}

function handleProcessingError(error, username) {
    console.error(`Erro geral ao processar séries para ${username}:`, error);
    const processingState = getProcessingState(username);
    processingState.isRunning = false;
    getIo().to(username).emit('processError', { message: error.message });
}
async function withRetry(operation, maxRetries = 3, delay = 2000) {
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
        try {
            return await operation();
        } catch (error) {
            if (attempt === maxRetries) throw error;
            
            const isTimeout = error.code === 'ECONNABORTED' || error.message.includes('timeout');
            const shouldRetry = isTimeout || (error.response && [408, 429, 500, 502, 503, 504].includes(error.response.status));
            
            if (!shouldRetry) throw error;
            
            console.log(`Attempt ${attempt} failed, retrying in ${delay/1000} seconds...`);
            await new Promise(resolve => setTimeout(resolve, delay));
        }
    }
}


module.exports = {
    getCategories: exports.getCategories,
    getBouquets: exports.getBouquets,
    processSeries: exports.processSeries,
    pauseProcess: exports.pauseProcess,
    resumeProcess: exports.resumeProcess,
    cancelProcess: exports.cancelProcess,
    getProcessStatus: exports.getProcessStatus,
       withRetry
};