const axios = require('axios');
const fs = require('fs').promises;
const path = require('path');
const { executeQuery, executeTransaction } = require('../utils/database');
const { updateUserProcess, removeUserProcess } = require('../socket');
const { getTMDbInfo } = require('../services/tmdb-filmes');

const userProcessingStates = new Map();
const DOWNLOAD_DIR = path.join(__dirname, '..', 'downloads');

const MAX_RETRIES = 3;
const INITIAL_RETRY_DELAY = 1000; // 1 second
const MAX_RETRY_DELAY = 5000; // 5 seconds

const BATCH_SIZE = 30;
const TMDB_RATE_LIMIT = 80; // Requests per 10 seconds
const TMDB_RATE_LIMIT_WINDOW = 10000; // 10 seconds in milliseconds

function initProcessingState(username) {
    return {
        isRunning: false,
        isPaused: false,
        isCompleted: false,
        processedItems: 0,
        addedItems: 0,
        duplicateMovies: 0,
        totalItems: 0,
        newMovies: [],
        log: [],
        startTime: null
    };
}

exports.getCategories = async (req, res) => {
    const { url } = req.body;
    try {
        const { username, password, server } = parseXtreamUrl(url);
        const [categoriesResponse, moviesCountResponse] = await Promise.all([
            axios.get(`${server}/player_api.php`, {
                params: { username, password, action: 'get_vod_categories' },
                timeout: 10000
            }),
            axios.get(`${server}/player_api.php`, {
                params: { username, password, action: 'get_vod_streams' },
                timeout: 10000
            })
        ]);

        const categories = categoriesResponse.data;
        const totalMovies = moviesCountResponse.data.length;

        res.json({ categories, totalMovies });
    } catch (error) {
        console.error('Error fetching categories:', error);
        res.status(500).json({ message: 'Error fetching categories', error: error.message });
    }
};

exports.getBouquets = async (req, res) => {
    const { username } = req.user;
    try {
        const bouquets = await executeQuery(username, 'SELECT id, bouquet_name FROM bouquets');
        if (!bouquets || bouquets.length === 0) {
            return res.status(404).json({ message: 'No bouquets found' });
        }
        res.json(bouquets);
    } catch (error) {
        console.error('Error fetching bouquets:', error);
        res.status(500).json({ message: 'Error fetching bouquets', error: error.message });
    }
};

exports.processMovies = async (req, res) => {
    const { url, bouquetId, selectedCategories } = req.body;
    const { username } = req.user;

    console.log(`Processing movies for user: ${username}`);
    console.log(`URL: ${url}`);
    console.log(`Bouquet ID: ${bouquetId}`);
    console.log(`Selected Categories: ${JSON.stringify(selectedCategories)}`);

    if (!selectedCategories || selectedCategories.length === 0) {
        return res.status(400).json({ message: 'No category selected' });
    }

    if (!userProcessingStates.has(username)) {
        userProcessingStates.set(username, initProcessingState(username));
    }

    const userState = userProcessingStates.get(username);

    if (userState.isRunning) {
        return res.status(400).json({ message: 'A processing task is already in progress for this user' });
    }

    userState.isRunning = true;
    userState.startTime = Date.now();

    res.json({ message: 'Processing started' });

    try {
        await processMoviesBackground(username, url, bouquetId, selectedCategories);
    } catch (error) {
        console.error(`Error in background processing for ${username}:`, error);
        updateUserProcess(username, { status: 'error', error: error.message });
    }
};

async function processMoviesBackground(username, url, bouquetId, selectedCategories) {
    const userState = userProcessingStates.get(username);
    try {
        const { username: xtreamUser, password: xtreamPass, server } = parseXtreamUrl(url);
        
        const userDownloadDir = path.join(DOWNLOAD_DIR, username, 'filmes');
        await fs.mkdir(userDownloadDir, { recursive: true });

        await downloadData(server, xtreamUser, xtreamPass, userDownloadDir);

        const categories = JSON.parse(await fs.readFile(path.join(userDownloadDir, 'categories.json'), 'utf8'));
        const movies = JSON.parse(await fs.readFile(path.join(userDownloadDir, 'movies.json'), 'utf8'));

        const categoriesToProcess = categories.filter(category => 
            selectedCategories.some(sc => sc.id === category.category_id && sc.selected)
        ).map(category => ({
            ...category,
            category_name: selectedCategories.find(sc => sc.id === category.category_id).newName || category.category_name
        }));

        await processCategories(username, categoriesToProcess);

        const moviesToProcess = movies.filter(movie => 
            selectedCategories.some(sc => sc.id === movie.category_id && sc.selected)
        );

        userState.totalItems = moviesToProcess.length;

        const bouquetResult = await executeQuery(username, 'SELECT bouquet_movies FROM bouquets WHERE id = ?', [bouquetId]);
        let currentBouquetMovies = bouquetResult[0]?.bouquet_movies ? JSON.parse(bouquetResult[0].bouquet_movies) : [];

        for (let i = 0; i < moviesToProcess.length; i += BATCH_SIZE) {
            if (!userState.isRunning) break;
            while (userState.isPaused) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }

            const batch = moviesToProcess.slice(i, i + BATCH_SIZE);
            await processBatch(username, batch, xtreamUser, xtreamPass, server, bouquetId, currentBouquetMovies, selectedCategories, userState);

            updateUserProcess(username, {
                status: 'processing',
                progress: (userState.processedItems / userState.totalItems) * 100,
                processedItems: userState.processedItems,
                addedItems: userState.addedItems,
                duplicateMovies: userState.duplicateMovies,
                newMovies: userState.newMovies.slice(-5),
                log: userState.log,
                timeRemaining: estimateTimeRemaining(userState),
                totalItems: userState.totalItems
            });
        }

        userState.isRunning = false;
        userState.isCompleted = true;
        updateUserProcess(username, {
            status: 'completed',
            progress: 100,
            processedItems: userState.processedItems,
            addedItems: userState.addedItems,
            duplicateMovies: userState.duplicateMovies,
            log: userState.log,
            totalItems: userState.totalItems
        });
    } catch (error) {
        console.error('Error processing movies:', error);
        userState.isRunning = false;
        updateUserProcess(username, { status: 'error', error: error.message });
    }
}



async function processCategories(username, categories) {
    for (const category of categories) {
        try {
            // Primeiro, verificamos se a categoria já existe
            const existingCategories = await executeQuery(
                username,
                'SELECT id, category_name FROM streams_categories WHERE id = ? AND category_type = ?',
                [category.category_id, 'movie']
            );

            if (Array.isArray(existingCategories) && existingCategories.length > 0) {
                const existingCategory = existingCategories[0];
                // Se a categoria existe e o nome é diferente, atualizamos
                if (existingCategory.category_name !== category.category_name) {
                    await executeQuery(
                        username,
                        'UPDATE streams_categories SET category_name = ? WHERE id = ? AND category_type = ?',
                        [category.category_name, category.category_id, 'movie']
                    );
                    console.log(`Updated category: ${category.category_name}`);
                } else {
                    console.log(`Category already exists and name is up to date: ${category.category_name}`);
                }
            } else {
                // Se a categoria não existe, inserimos
                await executeQuery(
                    username,
                    'INSERT INTO streams_categories (id, category_type, category_name, parent_id, cat_order) VALUES (?, ?, ?, ?, ?)',
                    [category.category_id, 'movie', category.category_name, 0, 0]
                );
                console.log(`Inserted new category: ${category.category_name}`);
            }
        } catch (error) {
            console.error(`Error processing category ${category.category_name}:`, error);
            // Registramos o erro, mas continuamos processando as outras categorias
        }
    }
}

async function downloadData(server, xtreamUser, xtreamPass, userDownloadDir) {
    const downloadFile = async (action, filename) => {
        const url = `${server}/player_api.php`;
        const filePath = path.join(userDownloadDir, filename);
        
        let retries = 0;
        while (retries < MAX_RETRIES) {
            try {
                const response = await axios.get(url, {
                    params: { username: xtreamUser, password: xtreamPass, action: action },
                    timeout: 60000,
                });

                if (!response.data || (Array.isArray(response.data) && response.data.length === 0)) {
                    throw new Error(`Received empty or invalid data for ${filename}`);
                }

                await fs.writeFile(filePath, JSON.stringify(response.data));
                console.log(`Successfully downloaded and saved ${filename}`);
                return;
            } catch (error) {
                retries++;
                console.error(`Error downloading ${filename} (attempt ${retries}):`, error.message);
                if (retries >= MAX_RETRIES) throw error;
                await new Promise(resolve => setTimeout(resolve, Math.min(INITIAL_RETRY_DELAY * Math.pow(2, retries), MAX_RETRY_DELAY)));
            }
        }
    };

    await Promise.all([
        downloadFile('get_vod_categories', 'categories.json'),
        downloadFile('get_vod_streams', 'movies.json')
    ]);
}

async function processBatch(username, batch, xtreamUser, xtreamPass, server, bouquetId, currentBouquetMovies, selectedCategories, userState) {
    const batchPromises = batch.map(async (movie) => {
        if (!userState.isRunning) return;
        while (userState.isPaused) {
            await new Promise(resolve => setTimeout(resolve, 1000));
        }

        try {
            const { name: movieTitle, year } = extractNameAndYear(movie.name);
            const { tmdbInfo, fromCache } = await getTMDbInfo(movieTitle, year);

            const streamUrl = `${server}/movie/${xtreamUser}/${xtreamPass}/${movie.stream_id}.${movie.container_extension}`;

            await executeTransaction(username, async (connection) => {
                const selectedCategory = selectedCategories.find(sc => sc.id === movie.category_id);
                const categoryId = selectedCategory ? selectedCategory.id : movie.category_id;

                // Verificar se o filme já existe na categoria atual
                const [existingMovie] = await connection.execute(
                    'SELECT id FROM streams WHERE stream_source = ? AND category_id = ?',
                    [JSON.stringify([streamUrl]), `[${categoryId}]`]
                );

                if (existingMovie.length > 0) {
                    // O filme já existe nesta categoria, então vamos apenas atualizar
                    const movieId = existingMovie[0].id;
                    await updateExistingMovie(connection, movieId, movie, tmdbInfo, streamUrl, categoryId);
                    updateProcessingState(userState, 'updated', movie.name, year, tmdbInfo?.id, fromCache);
                    console.log(`Filme atualizado: ${movie.name} na categoria ${categoryId}. TMDb ID: ${tmdbInfo?.id || 'N/A'} (${fromCache ? 'cache' : 'API'})`);
                } else {
                    // O filme não existe nesta categoria, então vamos inserir
                    const movieId = await insertNewMovie(connection, movie, tmdbInfo, streamUrl, categoryId);
                    updateProcessingState(userState, 'added', movie.name, year, tmdbInfo?.id, fromCache);
                    console.log(`Novo filme inserido: ${movie.name} na categoria ${categoryId}. TMDb ID: ${tmdbInfo?.id || 'N/A'} (${fromCache ? 'cache' : 'API'})`);

                    // Adicionar o novo filme ao bouquet se ainda não estiver lá
                    if (!currentBouquetMovies.includes(movieId)) {
                        currentBouquetMovies.push(movieId);
                        await connection.execute(
                            'UPDATE bouquets SET bouquet_movies = ? WHERE id = ?',
                            [JSON.stringify(currentBouquetMovies), bouquetId]
                        );
                    }
                }
            });
        } catch (error) {
            console.error(`Erro ao processar o filme ${movie.name}:`, error);
            userState.log.push(`Erro ao processar o filme ${movie.name}: ${error.message}`);
        }

        userState.processedItems++;
    });

    await Promise.all(batchPromises);
}

function updateProcessingState(state, type, title, year, tmdbId, fromCache) {
    const action = type === 'updated' ? 'Atualizado' : 'Adicionado';
    state[type === 'updated' ? 'duplicateMovies' : 'addedItems']++;
    const logMessage = `${action} filme: ${title} (${year || 'Ano desconhecido'}) - TMDb ID: ${tmdbId || 'N/A'} (${fromCache ? 'cache' : 'API'})`;
    state.log.push(logMessage);
    if (type === 'added') {
        state.newMovies.push(`${title} (${year || 'Ano desconhecido'})`);
    }
}

async function updateExistingMovie(connection, movieId, movie, tmdbInfo, streamUrl, categoryId) {
    const movieData = prepareMovieData(movie, tmdbInfo, streamUrl, movie.name, categoryId);
    const updateQuery = `UPDATE streams SET ${Object.keys(movieData).map(key => `${key} = ?`).join(', ')} WHERE id = ?`;
    await connection.execute(updateQuery, [...Object.values(movieData), movieId]);
}

async function insertNewMovie(connection, movie, tmdbInfo, streamUrl, categoryId) {
    const movieData = prepareMovieData(movie, tmdbInfo, streamUrl, movie.name, categoryId);
    const [result] = await connection.execute(
        `INSERT INTO streams (${Object.keys(movieData).join(', ')}) VALUES (${Object.keys(movieData).map(() => '?').join(', ')})`,
        Object.values(movieData)
    );
    return result.insertId;
}




function parseXtreamUrl(url) {
    try {
        const parsedUrl = new URL(url);
        return {
            username: parsedUrl.searchParams.get('username'),
            password: parsedUrl.searchParams.get('password'),
            server: `${parsedUrl.protocol}//${parsedUrl.hostname}:${parsedUrl.port}`
        };
    } catch (error) {
        throw new Error('Invalid Xtream URL format');
    }
}

function extractNameAndYear(fullName) {
    const yearMatch = fullName.match(/\b(19|20)\d{2}\b/);
    const year = yearMatch ? yearMatch[0] : null;
    const name = fullName.replace(/\b(19|20)\d{2}\b/, '').trim();
    return { name, year };
}
function estimateTimeRemaining(state) {
    const processedPerSecond = state.processedItems / ((Date.now() - state.startTime) / 1000);
    const remainingItems = state.totalItems - state.processedItems;
    return Math.round(remainingItems / processedPerSecond);
}

function prepareMovieData(movie, tmdbInfo, streamUrl, movieTitle, categoryId) {
    const movieProperties = {
        kinopoisk_url: tmdbInfo ? `https://www.themoviedb.org/movie/${tmdbInfo.id}` : '',
        tmdb_id: tmdbInfo ? tmdbInfo.id.toString() : '',
        name: tmdbInfo ? tmdbInfo.title : movie.name,
        o_name: tmdbInfo ? tmdbInfo.original_title : movie.name,
        cover_big: tmdbInfo ? `https://image.tmdb.org/t/p/w500${tmdbInfo.backdrop_path}` : '',
        movie_image: tmdbInfo ? `https://image.tmdb.org/t/p/w500${tmdbInfo.poster_path}` : movie.stream_icon,
        release_date: tmdbInfo ? tmdbInfo.release_date : '',
        episode_run_time: '',
        youtube_trailer: '',
        director: tmdbInfo ? tmdbInfo.director : '',
        actors: '',
        cast: tmdbInfo ? tmdbInfo.cast.join(', ') : '',
        description: '',
        plot: tmdbInfo ? tmdbInfo.overview : '',
        age: '',
        mpaa_rating: '',
        rating_count_kinopoisk: 0,
        country: '',
        genre: tmdbInfo ? tmdbInfo.genres.join(', ') : '',
        backdrop_path: tmdbInfo ? [`https://image.tmdb.org/t/p/w500${tmdbInfo.backdrop_path}`] : [],
        duration_secs: 0,
        duration: '00:00:00',
        video: [],
        audio: [],
        bitrate: 0,
        rating: tmdbInfo ? tmdbInfo.vote_average.toString() : '0'
    };

    return {
        type: 2,
        category_id: `[${categoryId}]`,
        stream_display_name: tmdbInfo ? tmdbInfo.title : movie.name,
        stream_source: JSON.stringify([streamUrl]),
        stream_icon: tmdbInfo ? `https://image.tmdb.org/t/p/w500${tmdbInfo.poster_path}` : movie.stream_icon,
        notes: null,
        enable_transcode: 0,
        transcode_attributes: null,
        custom_ffmpeg: null,
        movie_properties: JSON.stringify(movieProperties),
        movie_subtitles: null,
        read_native: 0,
        target_container: movie.container_extension,
        stream_all: 0,
        remove_subtitles: 0,
        custom_sid: null,
        epg_api: 0,
        epg_id: null,
        channel_id: null,
        epg_lang: null,
        '`order`': 0,
        auto_restart: null,
        transcode_profile_id: 0,
        gen_timestamps: 1,
        added: Math.floor(Date.now() / 1000),
        series_no: 0,
        direct_source: 1,
        tv_archive_duration: 0,
        tv_archive_server_id: 0,
        tv_archive_pid: 0,
        vframes_server_id: 0,
        vframes_pid: 0,
        movie_symlink: 0,
        rtmp_output: 0,
        allow_record: 0,
        probesize_ondemand: 128000,
        custom_map: null,
        external_push: null,
        delay_minutes: 0,
        tmdb_language: 'pt-br',
        llod: 0,
        year: tmdbInfo ? new Date(tmdbInfo.release_date).getFullYear() : (movie.year || null),
        rating: tmdbInfo ? tmdbInfo.vote_average : 0,
        plex_uuid: '',
        uuid: null,
        epg_offset: 0,
        updated: new Date().toISOString().slice(0, 19).replace('T', ' '),
        similar: null,
        tmdb_id: tmdbInfo ? tmdbInfo.id : null,
        adaptive_link: null,
        title_sync: null,
        fps_restart: 0,
        fps_threshold: 90
    };
}

exports.pauseProcess = (req, res) => {
    const { username } = req.user;
    const userState = userProcessingStates.get(username);
    if (!userState || !userState.isRunning) {
        return res.status(400).json({ message: 'No processing in progress for this user' });
    }
    userState.isPaused = true;
    res.json({ message: 'Processing paused' });
};

exports.resumeProcess = (req, res) => {
    const { username } = req.user;
    const userState = userProcessingStates.get(username);
    if (!userState || !userState.isRunning) {
        return res.status(400).json({ message: 'No processing in progress for this user' });
    }
    userState.isPaused = false;
    res.json({ message: 'Processing resumed' });
};

exports.cancelProcess = (req, res) => {
    const { username } = req.user;
    const userState = userProcessingStates.get(username);
    if (!userState || (!userState.isRunning && !userState.isCompleted)) {
        return res.status(400).json({ message: 'No processing in progress or completed for this user' });
    }
    userState.isRunning = false;
    userState.isPaused = false;
    userState.isCompleted = true;
    
    res.json({ 
        message: 'Processing finalized',
        finalResults: {
            totalProcessed: userState.processedItems,
            totalAdded: userState.addedItems,
            totalDuplicates: userState.duplicateMovies
        }
    });
    
    // Remove the user's processing state after a delay
    setTimeout(() => {
        userProcessingStates.delete(username);
        removeUserProcess(username);
    }, 5 * 60 * 1000);
};

exports.getProcessStatus = (req, res) => {
    const { username } = req.user;
    const userState = userProcessingStates.get(username);
    if (userState && (userState.isRunning || userState.isCompleted)) {
        res.json({
            isRunning: userState.isRunning,
            isPaused: userState.isPaused,
            isCompleted: userState.isCompleted,
            progress: {
                processedItems: userState.processedItems,
                addedItems: userState.addedItems,
                duplicateMovies: userState.duplicateMovies,
                timeRemaining: estimateTimeRemaining(userState),
                newMovies: userState.newMovies.slice(-5),
                totalItems: userState.totalItems,
                log: userState.log
            },
            finalResults: userState.isCompleted ? {
                totalProcessed: userState.processedItems,
                totalAdded: userState.addedItems,
                totalDuplicates: userState.duplicateMovies
            } : null
        });
    } else {
        res.json({ isRunning: false, isCompleted: false });
    }
};
exports.analyzeXtreamData = async (req, res) => {
    const { url } = req.body;
    const { username } = req.user;

    if (!url) {
        return res.status(400).json({ message: 'URL is required' });
    }

    try {
        const { username: xtreamUser, password: xtreamPass, server } = parseXtreamUrl(url);
        
        const userDownloadDir = path.join(DOWNLOAD_DIR, username, 'filmes');
        await fs.mkdir(userDownloadDir, { recursive: true });

        await downloadData(server, xtreamUser, xtreamPass, userDownloadDir);

        const categoriesData = await fs.readFile(path.join(userDownloadDir, 'categories.json'), 'utf8');
        const moviesData = await fs.readFile(path.join(userDownloadDir, 'movies.json'), 'utf8');

        const categories = JSON.parse(categoriesData);
        const movies = JSON.parse(moviesData);

        const categoryCounts = categories.map(category => ({
            ...category,
            movieCount: movies.filter(movie => movie.category_id === category.category_id).length
        }));

        res.json({ categories: categoryCounts, totalMovies: movies.length });
    } catch (error) {
        console.error('Error analyzing Xtream data:', error);
        res.status(500).json({ message: 'Error analyzing Xtream data', error: error.message });
    }
};
module.exports = {
    getCategories: exports.getCategories,
    getBouquets: exports.getBouquets,
    processMovies: exports.processMovies,
    pauseProcess: exports.pauseProcess,
    resumeProcess: exports.resumeProcess,
    cancelProcess: exports.cancelProcess,
    getProcessStatus: exports.getProcessStatus,
    analyzeXtreamData: exports.analyzeXtreamData  // Make sure this function is defined
};