const axios = require('axios');
const fs = require('fs').promises;
const path = require('path');
const { executeQuery, getUserConfig, configDbPool } = require('../utils/database');

const AUTOSYNC_DIR = path.join(__dirname, '..', 'autosync');
const DOWNLOAD_DIR = path.join(__dirname, '..', 'downloads');

exports.getUrls = async (req, res) => {
    try {
        const { username } = req.user;
        const userData = await getUserConfig(username);
        
        if (!userData) {
            return res.status(404).json({ message: 'User not found' });
        }

        res.json({
            movieUrl: userData.movie_url || '',
            seriesUrl: userData.series_url || ''
        });
    } catch (error) {
        console.error('Error fetching URLs:', error);
        res.status(500).json({ message: 'Error fetching URLs', error: error.message });
    }
};

exports.saveMovieUrl = async (req, res) => {
    try {
        const { username } = req.user;
        const { url } = req.body;
        
        const xtreamData = await testXtreamUrl(url, 'vod');
        const categories = await downloadCategories(url, 'movies');
        
        const userDir = path.join(AUTOSYNC_DIR, username);
        await fs.mkdir(path.join(userDir, 'filmes'), { recursive: true });
        
        const filePath = path.join(userDir, 'filmes', 'movies.json');
        await fs.writeFile(filePath, JSON.stringify(xtreamData));
        
        const categoriesPath = path.join(userDir, 'filmes', 'categories.json');
        await fs.writeFile(categoriesPath, JSON.stringify(categories));
        
        await configDbPool.execute(
            'UPDATE user_configs SET movie_url = ?, movie_count = ? WHERE username = ?',
            [url, xtreamData.length, username]
        );

        res.json({ message: 'Movie URL saved successfully', movieCount: xtreamData.length, categoriesCount: categories.length });
    } catch (error) {
        console.error('Error saving movie URL:', error);
        res.status(500).json({ message: 'Error saving movie URL', error: error.message });
    }
};

exports.saveSeriesUrl = async (req, res) => {
    try {
        const { username } = req.user;
        const { url } = req.body;
        
        const xtreamData = await testXtreamUrl(url, 'series');
        const categories = await downloadCategories(url, 'series');
        
        const userDir = path.join(AUTOSYNC_DIR, username);
        await fs.mkdir(path.join(userDir, 'series'), { recursive: true });
        
        const filePath = path.join(userDir, 'series', 'series.json');
        await fs.writeFile(filePath, JSON.stringify(xtreamData));
        
        const categoriesPath = path.join(userDir, 'series', 'categories.json');
        await fs.writeFile(categoriesPath, JSON.stringify(categories));
        
        await configDbPool.execute(
            'UPDATE user_configs SET series_url = ?, series_count = ? WHERE username = ?',
            [url, xtreamData.length, username]
        );

        res.json({ message: 'Series URL saved successfully', seriesCount: xtreamData.length, categoriesCount: categories.length });
    } catch (error) {
        console.error('Error saving series URL:', error);
        res.status(500).json({ message: 'Error saving series URL', error: error.message });
    }
};

async function testXtreamUrl(url, type) {
    try {
        const { username, password, server } = parseXtreamUrl(url);
        let action = type === 'vod' ? 'get_vod_streams' : 'get_series';
        
        const response = await axios.get(`${server}/player_api.php`, {
            params: { username, password, action },
            timeout: 30000
        });

        if (!response.data || (Array.isArray(response.data) && response.data.length === 0)) {
            throw new Error('Received empty or invalid data from Xtream server');
        }

        return response.data;
    } catch (error) {
        console.error('Error testing Xtream URL:', error);
        throw new Error('Failed to connect to Xtream server or invalid data received');
    }
}

async function downloadCategories(url, type) {
    try {
        const { username, password, server } = parseXtreamUrl(url);
        const action = type === 'movies' ? 'get_vod_categories' : 'get_series_categories';
        
        const response = await axios.get(`${server}/player_api.php`, {
            params: { username, password, action },
            timeout: 30000
        });

        if (!response.data || !Array.isArray(response.data)) {
            throw new Error('Received invalid category data from Xtream server');
        }

        return response.data;
    } catch (error) {
        console.error('Error downloading categories:', error);
        throw new Error('Failed to download categories from Xtream server');
    }
}

exports.getSyncLog = async (req, res) => {
    try {
        const { username } = req.user;
        const logFile = path.join(DOWNLOAD_DIR, username, 'synclog.json');
        
        const logData = await fs.readFile(logFile, 'utf8');
        const logs = JSON.parse(logData);
        
        // Filter for new movies and series
        const newContent = logs.filter(entry => {
            return entry.movies.some(movie => movie.type === 'new_movie') || 
                   entry.series.some(series => series.type === 'new_series' || series.type === 'new_episodes');
        });

        res.json(newContent);
    } catch (error) {
        console.error('Error fetching sync log:', error);
        if (error.code === 'ENOENT') {
            res.json({ message: 'No sync log found' });
        } else {
            res.status(500).json({ message: 'Error fetching sync log', error: error.message });
        }
    }
};

function parseXtreamUrl(url) {
    try {
        const parsedUrl = new URL(url);
        return {
            username: parsedUrl.searchParams.get('username'),
            password: parsedUrl.searchParams.get('password'),
            server: `${parsedUrl.protocol}//${parsedUrl.hostname}:${parsedUrl.port}`
        };
    } catch (error) {
        throw new Error('Invalid Xtream URL format');
    }
}

async function ensureAutoSyncDir() {
    try {
        await fs.mkdir(AUTOSYNC_DIR, { recursive: true });
    } catch (error) {
        console.error('Error creating AutoSync directory:', error);
        throw error;
    }
}

// Call this function when the server starts
ensureAutoSyncDir();

module.exports = {
    getUrls: exports.getUrls,
    saveMovieUrl: exports.saveMovieUrl,
    saveSeriesUrl: exports.saveSeriesUrl,
    getSyncLog: exports.getSyncLog
};