const mysql = require('mysql2/promise');
const fs = require('fs').promises;
const path = require('path');

const USERS_FILE = path.join(__dirname, '..', '..', 'users.json');

async function getConnection(username) {
  try {
    const usersData = await fs.readFile(USERS_FILE, 'utf8');
    const users = JSON.parse(usersData);
    const user = users[username];

    if (!user || !user.dbConfig) {
      throw new Error('User not found or database configuration missing');
    }

    const dbConfig = {
      ...user.dbConfig,
      database: 'xui'  // Sempre usar o banco de dados 'xui'
    };

    const connection = await mysql.createConnection(dbConfig);
    return connection;
  } catch (error) {
    console.error('Error creating database connection:', error);
    throw error;
  }
}

async function executeQuery(username, query, params = []) {
  let connection;
  try {
    connection = await getConnection(username);
    const [results] = await connection.execute(query, params);
    return results;
  } catch (error) {
    console.error('Database query error:', error);
    throw error;
  } finally {
    if (connection) await connection.end();
  }
}

async function executeTransaction(username, queries) {
  let connection;
  try {
    connection = await getConnection(username);
    await connection.beginTransaction();

    const results = [];
    for (const [query, params] of queries) {
      const [result] = await connection.execute(query, params);
      results.push(result);
    }

    await connection.commit();
    return results;
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Transaction error:', error);
    throw error;
  } finally {
    if (connection) await connection.end();
  }
}

module.exports = {
  executeQuery,
  executeTransaction
};