const cron = require('node-cron');
const path = require('path');
const fs = require('fs').promises;
const axios = require('axios');
const { configDbPool, executeQuery, executeTransaction } = require('./utils/database');
const { processSeriesItem, processEpisodes } = require('./controllers/seriesController');
const { getTMDbInfo: getTMDbSeriesInfo } = require('./services/tmdb-series');

const AUTOSYNC_DIR = path.join(__dirname, 'autosync');
const DEBUG = process.env.DEBUG_AUTO_SYNC === 'true';

async function autoSyncContent() {
    console.log('Starting auto-sync process');
    try {
        const users = await getUsersWithUrls();

        for (const user of users) {
            await syncUserContent(user);
        }

        console.log('Auto-sync process completed');
    } catch (error) {
        console.error('Error in auto-sync process:', error);
    }
}

async function getUsersWithUrls() {
    const [rows] = await configDbPool.execute('SELECT username, series_url FROM user_configs WHERE series_url IS NOT NULL AND series_url != ""');
    return rows;
}

async function syncUserContent(user) {
    console.log(`Syncing content for user: ${user.username}`);
    let syncSummary = { newSeries: 0, newEpisodes: 0 };

    try {
        if (user.series_url) {
            console.log(`Syncing series for user: ${user.username}`);
            syncSummary = await syncSeries(user);
        }

        console.log(`Sync summary for ${user.username}:`);
        console.log(`New series: ${syncSummary.newSeries}`);
        console.log(`New episodes: ${syncSummary.newEpisodes}`);
    } catch (error) {
        console.error(`Error syncing content for ${user.username}:`, error);
    }
}

async function syncSeries(user) {
    const userDir = path.join(AUTOSYNC_DIR, user.username, 'series');
    const seriesFile = path.join(userDir, 'series.json');
    const categoriesFile = path.join(userDir, 'categories.json');

    await fs.mkdir(userDir, { recursive: true });

    const onlineSeriesData = await downloadSeriesData(user.series_url);
    let localSeriesData = await loadLocalSeriesData(seriesFile);

    const categories = await loadCategories(categoriesFile, user.series_url);

    let newSeriesCount = 0;
    let newEpisodesCount = 0;
    let updatedLocalSeriesData = [];

    for (const onlineSeries of onlineSeriesData) {
        const localSeries = localSeriesData.find(s => s.series_id === onlineSeries.series_id);
        
        try {
            if (!localSeries) {
                // New series
                const result = await processNewSeries(user, onlineSeries, categories);
                newSeriesCount += result.newSeries;
                newEpisodesCount += result.newEpisodes;
                updatedLocalSeriesData.push(onlineSeries);
            } else {
                // Existing series, check for new episodes
                const newEpisodes = await processExistingSeries(user, localSeries, onlineSeries);
                newEpisodesCount += newEpisodes;
                // Update local series data with any changes
                updatedLocalSeriesData.push({...localSeries, ...onlineSeries});
            }
        } catch (error) {
            console.error(`Error processing series ${onlineSeries.name}:`, error);
        }
    }

    // Update the local series.json file
    await fs.writeFile(seriesFile, JSON.stringify(updatedLocalSeriesData, null, 2));

    return { newSeries: newSeriesCount, newEpisodes: newEpisodesCount };
}

async function loadLocalSeriesData(seriesFile) {
    try {
        const data = await fs.readFile(seriesFile, 'utf8');
        return JSON.parse(data);
    } catch (error) {
        console.log(`No existing series file found, creating a new one`);
        return [];
    }
}

async function loadCategories(categoriesFile, seriesUrl) {
    try {
        const data = await fs.readFile(categoriesFile, 'utf8');
        return JSON.parse(data);
    } catch (error) {
        console.log('Categories file not found, downloading categories...');
        const categories = await downloadCategories(seriesUrl);
        await fs.writeFile(categoriesFile, JSON.stringify(categories, null, 2));
        return categories;
    }
}

async function downloadCategories(url) {
    const { username, password, server } = parseXtreamUrl(url);
    const response = await axios.get(`${server}/player_api.php`, {
        params: { username, password, action: 'get_series_categories' },
        timeout: 30000
    });
    return response.data;
}

async function processNewSeries(user, series, categories) {
    const category = categories.find(cat => cat.category_id === series.category_id);
    if (!category) {
        console.log(`Category not found for series: ${series.name}`);
        return { newSeries: 0, newEpisodes: 0 };
    }

    console.log(`Processing new series: ${series.name}`);
    const result = await processSeries(user.username, series, user.series_url, category);
    console.log(`Added new series: ${series.name} with ${result.episodeCount} episodes`);
    return { newSeries: 1, newEpisodes: result.episodeCount };
}

async function processExistingSeries(user, localSeries, onlineSeries) {
    console.log(`Checking for new episodes in series: ${onlineSeries.name}`);
    
    try {
        // Check if the series exists in the database
        const seriesExists = await checkSeriesExistsInDatabase(user.username, onlineSeries.name);
        if (!seriesExists) {
            console.log(`Series ${onlineSeries.name} not found in database. Reprocessing as new series.`);
            const category = await getCategoryForSeries(user.username, onlineSeries.category_id);
            if (category) {
                const result = await processSeries(user.username, onlineSeries, user.series_url, category);
                return result.episodeCount;
            } else {
                console.log(`Category not found for series: ${onlineSeries.name}. Skipping.`);
                return 0;
            }
        }

        const localEpisodes = localSeries.episodes || [];
        const onlineEpisodes = await getSeriesEpisodes(user.series_url, onlineSeries.series_id);
        
        let newEpisodesCount = 0;

        for (const onlineEpisode of onlineEpisodes) {
            const existingEpisode = localEpisodes.find(e => 
                e.season == onlineEpisode.season && e.episode_num == onlineEpisode.episode_num
            );

            if (!existingEpisode) {
                await processNewEpisode(user.username, onlineSeries, onlineEpisode, user.series_url);
                newEpisodesCount++;
            }
        }

        if (newEpisodesCount > 0) {
            console.log(`Added ${newEpisodesCount} new episodes to series: ${onlineSeries.name}`);
        }

        return newEpisodesCount;
    } catch (error) {
        console.error(`Error processing existing series ${onlineSeries.name}:`, error);
        return 0;
    }
}

async function checkSeriesExistsInDatabase(username, seriesName) {
    try {
        const result = await executeQuery(username, 'SELECT COUNT(*) as count FROM streams_series WHERE title = ?', [seriesName]);
        return result[0] && result[0].count > 0;
    } catch (error) {
        console.error(`Error checking if series ${seriesName} exists in database:`, error);
        return false;
    }
}

async function getCategoryForSeries(username, categoryId) {
    try {
        const result = await executeQuery(username, 'SELECT * FROM streams_categories WHERE category_id = ?', [categoryId]);
        return result[0] || null;
    } catch (error) {
        console.error(`Error getting category for categoryId ${categoryId}:`, error);
        return null;
    }
}

async function processSeries(username, series, seriesUrl, category) {
    const { username: xtreamUser, password: xtreamPass, server } = parseXtreamUrl(seriesUrl);
    const tmdbInfo = await getTMDbSeriesInfo(series.name);

    return await executeTransaction(username, async (connection) => {
        const seriesResult = await processSeriesItem(connection, series, tmdbInfo, xtreamUser, xtreamPass, server, category);
        const episodeStats = await processEpisodes(connection, seriesResult.seriesId, series, xtreamUser, xtreamPass, server, username, category);
        return { ...seriesResult, ...episodeStats };
    });
}

async function processNewEpisode(username, series, episode, seriesUrl) {
    const { username: xtreamUser, password: xtreamPass, server } = parseXtreamUrl(seriesUrl);
    
    return await executeTransaction(username, async (connection) => {
        const [existingSeriesDb] = await connection.query(
            'SELECT id FROM streams_series WHERE title = ?',
            [series.name]
        );

        if (existingSeriesDb.length === 0) {
            console.log(`Series ${series.name} not found in database, skipping episode`);
            return;
        }

        const seriesId = existingSeriesDb[0].id;
        const category = await getCategoryForSeries(username, series.category_id);

        if (!category) {
            console.log(`Category not found for series ${series.name}, skipping episode`);
            return;
        }

        await processEpisodes(connection, seriesId, {
            ...series,
            episodes: [episode]
        }, xtreamUser, xtreamPass, server, username, category);

        console.log(`Added new episode: S${episode.season}E${episode.episode_num} for series ${series.name}`);
    });
}

async function getSeriesEpisodes(seriesUrl, seriesId) {
    const { username, password, server } = parseXtreamUrl(seriesUrl);
    const response = await axios.get(`${server}/player_api.php`, {
        params: { username, password, action: 'get_series_info', series_id: seriesId },
        timeout: 30000
    });
    return Object.values(response.data.episodes).flat();
}

async function downloadSeriesData(url) {
    const { username, password, server } = parseXtreamUrl(url);
    const response = await axios.get(`${server}/player_api.php`, {
        params: { username, password, action: 'get_series' },
        timeout: 30000
    });
    return Object.values(response.data);
}

function parseXtreamUrl(url) {
    const parsedUrl = new URL(url);
    return {
        username: parsedUrl.searchParams.get('username'),
        password: parsedUrl.searchParams.get('password'),
        server: `${parsedUrl.protocol}//${parsedUrl.hostname}:${parsedUrl.port}`
    };
}

// Schedule the auto-sync to run every 6 hours
cron.schedule('0 */6 * * *', () => {
    autoSyncContent();
});

// If DEBUG is true, run the auto-sync immediately
if (DEBUG) {
    console.log('Running auto-sync in debug mode');
    autoSyncContent();
}

module.exports = { autoSyncContent };