const express = require('express');
const cors = require('cors');
const dotenv = require('dotenv');
const http = require('http');
const authRoutes = require('./routes/auth');
const movieRoutes = require('./routes/movies');
const seriesRoutes = require('./routes/series');
const homeRoutes = require('./routes/home');
const autosyncRoutes = require('./routes/autosyncroutes');
const { initializeSocket } = require('./socket');
const morgan = require('morgan');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const timeout = require('connect-timeout');
const path = require('path');
const { autoSyncContent } = require('./autoSyncScheduler');

dotenv.config();
const app = express();
const server = http.createServer(app);

// Inicializar Socket.IO com configurações de reconexão
const io = initializeSocket(server, {
  pingTimeout: 60000,
  pingInterval: 25000,
  reconnection: true,
  reconnectionAttempts: Infinity,
  reconnectionDelay: 1000,
  reconnectionDelayMax: 5000,
});

// Middleware de segurança e log
app.use(helmet());
app.use(morgan('combined'));

// Configuração do CORS
app.use(cors({
  origin: 'http://sync.zpainel.site:3000',
  credentials: true
}));

// Limite de requisições
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 100,
  message: 'Too many requests from this IP, please try again after 15 minutes'
});
app.use(limiter);
app.use(express.json());

// Timeout para requisições
app.use(timeout('30s'));

// Diretório para arquivos estáticos (incluindo a pasta autosync)
app.use('/autosync', express.static(path.join(__dirname, 'autosync')));

// Tratamento de eventos do Socket.IO
io.on('connection', (socket) => {
  console.log(`New client connected: ${socket.id}`);
  socket.on('disconnect', (reason) => {
    console.log(`Client disconnected: ${socket.id}, reason: ${reason}`);
    if (reason === 'io server disconnect') {
      // Se o servidor desconectou, tentar reconectar
      socket.connect();
    }
  });
  socket.on('error', (error) => {
    console.error(`Socket error for ${socket.id}:`, error);
  });
});

// Middleware para disponibilizar io para as rotas
app.use((req, res, next) => {
  req.io = io;
  next();
});

// Rotas
app.use('/api/auth', authRoutes);
app.use('/api/movies', movieRoutes);
app.use('/api/series', seriesRoutes);
app.use('/api/home', homeRoutes);
app.use('/api/autosync', autosyncRoutes);

// Tratamento de erros globais
app.use((err, req, res, next) => {
  console.error('Erro detectado:', err.stack);
  if (req.timedout) {
    return res.status(503).send('Requisição demorou muito tempo para ser processada, tente novamente mais tarde.');
  }
  res.status(500).send('Algo deu errado! Estamos trabalhando para corrigir.');
});

// Tratamento de erros do servidor
server.on('error', (error) => {
  console.error('Erro no servidor:', error);
  if (error.code === 'EADDRINUSE') {
    console.error('Porta já em uso, tentando reiniciar...');
    setTimeout(() => {
      server.close();
      server.listen(PORT);
    }, 1000);
  } else {
    process.exit(1);
  }
});

const PORT = process.env.PORT || 5000;
server.listen(PORT, () => console.log(`Server running on port ${PORT}`));

// Iniciar o processo de auto-sincronização


module.exports = app;