import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { FaSave, FaFilm, FaTv, FaSync } from 'react-icons/fa';

const BACKEND_URL = 'http://sync.zpainel.site:5000';

const AutoSyncPage = () => {
  const [movieUrl, setMovieUrl] = useState('');
  const [seriesUrl, setSeriesUrl] = useState('');
  const [isLoadingMovies, setIsLoadingMovies] = useState(false);
  const [isLoadingSeries, setIsLoadingSeries] = useState(false);
  const [movieCount, setMovieCount] = useState(0);
  const [seriesCount, setSeriesCount] = useState(0);
  const [error, setError] = useState('');
  const [syncLog, setSyncLog] = useState(null);

  useEffect(() => {
    fetchExistingUrls();
    fetchSyncLog();
  }, []);

  const fetchExistingUrls = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/autosync/urls`);
      setMovieUrl(response.data.movieUrl || '');
      setSeriesUrl(response.data.seriesUrl || '');
      setMovieCount(response.data.movieCount || 0);
      setSeriesCount(response.data.seriesCount || 0);
    } catch (err) {
      setError('Erro ao buscar URLs existentes');
      console.error(err);
    }
  };

  const fetchSyncLog = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/autosync/log`);
      setSyncLog(response.data);
    } catch (err) {
      console.error('Erro ao buscar log de sincronização:', err);
    }
  };

  const handleSaveMovieUrl = async () => {
    setIsLoadingMovies(true);
    setError('');
    try {
      const response = await axios.post(`${BACKEND_URL}/api/autosync/movies`, { url: movieUrl });
      setMovieCount(response.data.movieCount);
      alert('URL de filmes salva com sucesso!');
    } catch (err) {
      setError('Erro ao salvar URL de filmes');
      console.error(err);
    } finally {
      setIsLoadingMovies(false);
    }
  };

  const handleSaveSeriesUrl = async () => {
    setIsLoadingSeries(true);
    setError('');
    try {
      const response = await axios.post(`${BACKEND_URL}/api/autosync/series`, { url: seriesUrl });
      setSeriesCount(response.data.seriesCount);
      alert('URL de séries salva com sucesso!');
    } catch (err) {
      setError('Erro ao salvar URL de séries');
      console.error(err);
    } finally {
      setIsLoadingSeries(false);
    }
  };

  return (
    <div className="movie-page">
      <h1 className="movie-page__title">AutoSync</h1>
      
      {error && <p className="movie-page__error">{error}</p>}

      <div className="movie-page__card">
        <h2 className="movie-page__card-title">URL de Filmes</h2>
        <div className="movie-page__form-group">
          <label htmlFor="movie-url">
            <FaFilm /> URL de Filmes:
          </label>
          <input
            type="text"
            id="movie-url"
            value={movieUrl}
            onChange={(e) => setMovieUrl(e.target.value)}
            placeholder="http://exemplo.com:porta/..."
          />
        </div>
        <button 
          onClick={handleSaveMovieUrl} 
          disabled={isLoadingMovies}
          className="movie-page__btn movie-page__btn--primary"
        >
          <FaSave /> {isLoadingMovies ? 'Salvando...' : 'Salvar'}
        </button>
        {movieCount > 0 && (
          <p className="movie-page__stats-card">Total de filmes: {movieCount}</p>
        )}
      </div>

      <div className="movie-page__card">
        <h2 className="movie-page__card-title">URL de Séries</h2>
        <div className="movie-page__form-group">
          <label htmlFor="series-url">
            <FaTv /> URL de Séries:
          </label>
          <input
            type="text"
            id="series-url"
            value={seriesUrl}
            onChange={(e) => setSeriesUrl(e.target.value)}
            placeholder="http://exemplo.com:porta/..."
          />
        </div>
        <button 
          onClick={handleSaveSeriesUrl} 
          disabled={isLoadingSeries}
          className="movie-page__btn movie-page__btn--primary"
        >
          <FaSave /> {isLoadingSeries ? 'Salvando...' : 'Salvar'}
        </button>
        {seriesCount > 0 && (
          <p className="movie-page__stats-card">Total de séries: {seriesCount}</p>
        )}
      </div>

      {syncLog && (
        <div className="movie-page__card">
          <h2 className="movie-page__card-title">
            <FaSync /> Log de Sincronização
          </h2>
          <pre className="movie-page__log">
            {JSON.stringify(syncLog, null, 2)}
          </pre>
        </div>
      )}
    </div>
  );
};

export default AutoSyncPage;