import React, { useState, useEffect, useRef } from 'react';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';
import io from 'socket.io-client';
import { FaPlay, FaPause, FaStop, FaCopy, FaTv, FaCheckCircle, FaExclamationTriangle, FaCheck, FaTimes } from 'react-icons/fa';

const BACKEND_URL = 'http://sync.zpainel.site:5000';

const SeriesPage = () => {
  const [url, setUrl] = useState('');
  const [categories, setCategories] = useState([]);
  const [totalSeries, setTotalSeries] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState('');
  const [processState, setProcessState] = useState('idle');
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [progress, setProgress] = useState({
    processedItems: 0,
    addedItems: 0,
    duplicateSeries: 0,
    timeRemaining: 0,
    newSeries: [],
    totalItems: 0,
    totalEpisodes: 0,
    processedEpisodes: 0,
    addedEpisodes: 0,
    duplicateEpisodes: 0
  });
  const [bouquets, setBouquets] = useState([]);
  const [selectedBouquet, setSelectedBouquet] = useState('');
  const [logMessages, setLogMessages] = useState([]);
  const [allSelected, setAllSelected] = useState(false);
  const [showSuccessMessage, setShowSuccessMessage] = useState(false);
  const navigate = useNavigate();
  const socketRef = useRef(null);

  useEffect(() => {
    const token = localStorage.getItem('token');
    if (token) {
      setIsAuthenticated(true);
      axios.defaults.headers.common['Authorization'] = `Bearer ${token}`;
      fetchBouquets();
      checkProcessStatus();
    } else {
      navigate('/login');
    }
  }, [navigate]);

  useEffect(() => {
    if (isAuthenticated) {
      socketRef.current = io(BACKEND_URL, {
        query: { token: localStorage.getItem('token') }
      });

      socketRef.current.on('connect', () => {
        console.log('Socket connected');
      });

      socketRef.current.on('connect_error', (error) => {
        console.error('Socket connection error:', error);
      });

      socketRef.current.on('processUpdate', handleProcessUpdate);
      socketRef.current.on('processComplete', handleProcessComplete);
      socketRef.current.on('processError', handleProcessError);

      return () => {
        if (socketRef.current) {
          socketRef.current.disconnect();
        }
      };
    }
  }, [isAuthenticated]);

  const fetchBouquets = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/series/bouquets`);
      if (response.data && response.data.length > 0) {
        setBouquets(response.data);
        setSelectedBouquet(response.data[0].id);
      } else {
        setError('Nenhum bouquet encontrado. Por favor, crie um bouquet primeiro.');
      }
    } catch (err) {
      console.error('Error fetching bouquets:', err);
      handleError(err);
    }
  };

  const checkProcessStatus = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/series/process-status`);
      if (response.data.isRunning) {
        setProcessState(response.data.isPaused ? 'paused' : 'running');
        setProgress(response.data.progress);
      } else if (response.data.isCompleted) {
        setProcessState('completed');
        setProgress(response.data.progress);
        setShowSuccessMessage(true);
      }
    } catch (err) {
      console.error('Error checking process status:', err);
    }
  };

  const handleAnalyze = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError('');
    setCategories([]);
    setTotalSeries(0);
    setProcessState('idle');

    try {
      const response = await axios.post(`${BACKEND_URL}/api/series/categories`, { url });
      setCategories(response.data.categories.map(category => ({
        ...category,
        selected: false,
        newName: category.category_name
      })));
      setTotalSeries(response.data.totalSeries);
    } catch (err) {
      handleError(err);
    } finally {
      setIsLoading(false);
    }
  };

  const handleCategorySelect = (categoryId) => {
    setCategories(categories.map(category =>
      category.category_id === categoryId ? { ...category, selected: !category.selected } : category
    ));
    updateAllSelectedState();
  };

  const handleCategoryRename = (categoryId, newName) => {
    setCategories(categories.map(category =>
      category.category_id === categoryId ? { ...category, newName } : category
    ));
  };

  const handleSelectAllCategories = () => {
    setCategories(categories.map(category => ({ ...category, selected: true })));
    setAllSelected(true);
  };

  const handleDeselectAllCategories = () => {
    setCategories(categories.map(category => ({ ...category, selected: false })));
    setAllSelected(false);
  };

  const updateAllSelectedState = () => {
    setAllSelected(categories.every(category => category.selected));
  };

  const handleStartProcess = async () => {
    setIsLoading(true);
    setError('');
    setProcessState('running');
    setProgress({
      processedItems: 0,
      addedItems: 0,
      duplicateSeries: 0,
      timeRemaining: 0,
      newSeries: [],
      totalItems: 0,
      totalEpisodes: 0,
      processedEpisodes: 0,
      addedEpisodes: 0,
      duplicateEpisodes: 0
    });
    setLogMessages([]);

    const selectedCategories = categories
      .filter(category => category.selected)
      .map(({ category_id, newName }) => ({
        id: category_id,
        newName,
        selected: true
      }));

    console.log('Selected categories:', selectedCategories);

    if (selectedCategories.length === 0) {
      setError('Por favor, selecione pelo menos uma categoria.');
      setIsLoading(false);
      setProcessState('idle');
      return;
    }

    try {
      await axios.post(`${BACKEND_URL}/api/series/process`, {
        url,
        bouquetId: selectedBouquet,
        selectedCategories
      });
    } catch (err) {
      handleError(err);
      setProcessState('idle');
    } finally {
      setIsLoading(false);
    }
  };

  const handlePauseProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/series/pause`);
      setProcessState('paused');
    } catch (err) {
      handleError(err);
    }
  };

  const handleResumeProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/series/resume`);
      setProcessState('running');
    } catch (err) {
      handleError(err);
    }
  };

  const handleCancelProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/series/cancel`);
      setProcessState('idle');
      setProgress({
        processedItems: 0,
        addedItems: 0,
        duplicateSeries: 0,
        timeRemaining: 0,
        newSeries: [],
        totalItems: 0,
        totalEpisodes: 0,
        processedEpisodes: 0,
        addedEpisodes: 0,
        duplicateEpisodes: 0
      });
      setLogMessages([]);
      setShowSuccessMessage(false);
      setUrl('');
      setCategories([]);
      setTotalSeries(0);
    } catch (err) {
      handleError(err);
    }
  };

  const handleProcessUpdate = (data) => {
    setProgress(prevProgress => ({
      ...prevProgress,
      ...data
    }));
    
    if (data.newSeries && Array.isArray(data.newSeries)) {
      setLogMessages(prevLogs => [...prevLogs, ...data.newSeries]);
    } else if (data.log) {
      let newLogs = [];
      if (Array.isArray(data.log)) {
        newLogs = data.log;
      } else if (typeof data.log === 'string') {
        newLogs = [data.log];
      } else if (typeof data.log === 'object') {
        newLogs = [JSON.stringify(data.log)];
      }
      setLogMessages(prevLogs => [...prevLogs, ...newLogs]);
    }
  };

  const handleProcessComplete = (data) => {
    setProcessState('completed');
    setProgress(prevProgress => ({
      ...prevProgress,
      ...data
    }));
    setShowSuccessMessage(true);
    
    if (data.completedSeries && Array.isArray(data.completedSeries)) {
      setLogMessages(prevLogs => [...prevLogs, ...data.completedSeries]);
    } else if (data.log) {
      let newLogs = [];
      if (Array.isArray(data.log)) {
        newLogs = data.log;
      } else if (typeof data.log === 'string') {
        newLogs = [data.log];
      } else if (typeof data.log === 'object') {
        newLogs = [JSON.stringify(data.log)];
      }
      setLogMessages(prevLogs => [...prevLogs, ...newLogs]);
    }
  };

  const handleProcessError = (error) => {
    console.error('Erro no processamento:', error);
    setError(`Erro no processamento: ${error.message}`);
    setProcessState('idle');
  };

  const handleError = (err) => {
    if (err.response && err.response.status === 401) {
      setError('Sessão expirada. Por favor, faça login novamente.');
      setIsAuthenticated(false);
      localStorage.removeItem('token');
      navigate('/login');
    } else {
      setError(err.response?.data?.message || err.message || 'Ocorreu um erro');
    }
  };

  const copyLogToClipboard = () => {
    const logText = logMessages.join('\n');
    
    if (navigator.clipboard && navigator.clipboard.writeText) {
      navigator.clipboard.writeText(logText)
        .then(() => alert('Log copiado para a área de transferência!'))
        .catch(err => {
          console.error('Falha ao copiar log:', err);
          fallbackCopyTextToClipboard(logText);
        });
    } else {
      fallbackCopyTextToClipboard(logText);
    }
  };

  const fallbackCopyTextToClipboard = (text) => {
    const textArea = document.createElement("textarea");
    textArea.value = text;
    
    textArea.style.top = "0";
    textArea.style.left = "0";
    textArea.style.position = "fixed";

    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();

    try {
      const successful = document.execCommand('copy');
      const msg = successful ? 'Log copiado para a área de transferência!' : 'Falha ao copiar o log.';
      alert(msg);
    } catch (err) {
      console.error('Fallback: Oops, unable to copy', err);
      alert('Falha ao copiar o log. Por favor, tente copiar manualmente.');
    }

    document.body.removeChild(textArea);
  };

  return (
    <div className="movie-page">
      <h1 className="movie-page__title">Sync Séries</h1>
      
      {(processState === 'idle' || (processState === 'completed' && !showSuccessMessage)) && (
        <div className="movie-page__card">
          <h2 className="movie-page__card-title">Analisar Fonte</h2>
          <form onSubmit={handleAnalyze} className="movie-page__form">
            <div className="movie-page__form-group">
              <label htmlFor="xtream-url">Insira a URL da Fonte:</label>
              <input
                type="text"
                id="xtream-url"
                value={url}
                onChange={(e) => setUrl(e.target.value)}
                required
                placeholder="http://exemplo.com:porta/..."
              />
            </div>
            <button 
              type="submit" 
              disabled={isLoading}
              className="movie-page__btn movie-page__btn--primary"
            >
              {isLoading ? 'Processando...' : 'Analisar'}
            </button>
          </form>
        </div>
      )}

      {error && <p className="movie-page__error">{error}</p>}

      {categories.length > 0 && processState === 'idle' && (
        <div className="movie-page__card">
          <h2 className="movie-page__card-title">Categorias</h2>
          <div className="movie-page__category-controls">
            <button 
              onClick={handleSelectAllCategories}
              className="movie-page__btn movie-page__btn--secondary"
            >
              <FaCheck /> Selecionar Todas
            </button>
            <button 
              onClick={handleDeselectAllCategories}
              className="movie-page__btn movie-page__btn--secondary"
            >
              <FaTimes /> Desmarcar Todas
            </button>
          </div>
   <div className="movie-page__categories-list">
  {categories.map(category => (
    <div key={category.category_id} className="movie-page__category-item">
      <input
        type="checkbox"
        id={`category-${category.category_id}`}
        checked={category.selected}
        onChange={() => handleCategorySelect(category.category_id)}
      />
      <input
        type="text"
        value={category.newName}
        onChange={(e) => handleCategoryRename(category.category_id, e.target.value)}
      />
      <span>{category.series_count} séries</span>
    </div>
  ))}
</div>
          
         <div className="movie-page__stats-grid">
  <div className="movie-page__stats-card">
    <h3>Total de categorias</h3>
    <p>{categories.length}</p>
  </div>
  <div className="movie-page__stats-card">
    <h3>Total de séries</h3>
    <p>{totalSeries}</p>
  </div>
</div>

{bouquets.length > 0 ? (
  <div className="movie-page__form-group">
    <label htmlFor="bouquet-select">Selecione o Bouquet:</label>
    <select
      id="bouquet-select"
      value={selectedBouquet}
      onChange={(e) => setSelectedBouquet(e.target.value)}
      required
    >
      {bouquets.map((bouquet) => (
        <option key={bouquet.id} value={bouquet.id}>
          {bouquet.bouquet_name}
        </option>
      ))}
    </select>
  </div>
) : (
  <p className="movie-page__error">Nenhum bouquet disponível. Por favor, crie um bouquet primeiro.</p>
)}

<button 
  onClick={handleStartProcess} 
  disabled={isLoading || processState !== 'idle' || !selectedBouquet || categories.filter(c => c.selected).length === 0}
  className="movie-page__btn movie-page__btn--success"
>
  <FaPlay /> Iniciar Processamento
</button>
</div>
)}

{(processState === 'running' || processState === 'paused') && (
<div className="movie-page__card">
  <h2 className="movie-page__card-title">Estatísticas</h2>
  <div className="movie-page__progress-bar">
    <div 
      className="movie-page__progress"
      style={{width: `${(progress.processedItems / progress.totalItems) * 100}%`}}
    ></div>
  </div>
  <div className="movie-page__stats-grid">
    <div className="movie-page__stats-card">
      <FaTv />
      <h3>Séries Processadas</h3>
      <p>{progress.processedItems} / {progress.totalItems}</p>
    </div>
    <div className="movie-page__stats-card">
      <FaCheckCircle />
      <h3>Séries Adicionadas</h3>
      <p>{progress.addedItems}</p>
    </div>
    <div className="movie-page__stats-card">
      <FaExclamationTriangle />
      <h3>Séries Duplicadas</h3>
      <p>{progress.duplicateSeries}</p>
    </div>
    <div className="movie-page__stats-card">
      <h3>Tempo Restante Estimado</h3>
      <p>{progress.timeRemaining || 'Calculando...'}</p>
    </div>
  </div>
  <div className="movie-page__stats-grid">
    <div className="movie-page__stats-card">
      <h3>Total de Episódios</h3>
      <p>{progress.totalEpisodes}</p>
    </div>
    <div className="movie-page__stats-card">
      <h3>Episódios Processados</h3>
      <p>{progress.processedEpisodes}</p>
    </div>
    <div className="movie-page__stats-card">
      <h3>Episódios Adicionados</h3>
      <p>{progress.addedEpisodes}</p>
    </div>
    <div className="movie-page__stats-card">
      <h3>Episódios Duplicados</h3>
      <p>{progress.duplicateEpisodes}</p>
    </div>
  </div>
  <div className="movie-page__log-container">
    <h3 className="movie-page__subtitle">Log de Processamento:</h3>
    <ul className="movie-page__log-list">
      {logMessages.map((message, index) => (
        <li key={index}>{message}</li>
      ))}
    </ul>
  </div>
  <div className="movie-page__button-group">
    {processState === 'running' && (
      <button onClick={handlePauseProcess} className="movie-page__btn movie-page__btn--warning">
        <FaPause /> Pausar
      </button>
    )}
    {processState === 'paused' && (
      <button onClick={handleResumeProcess} className="movie-page__btn movie-page__btn--success">
        <FaPlay /> Continuar
      </button>
    )}
    <button onClick={handleCancelProcess} className="movie-page__btn movie-page__btn--danger">
      <FaStop /> Cancelar
    </button>
    <button onClick={copyLogToClipboard} className="movie-page__btn movie-page__btn--secondary">
      <FaCopy /> Copiar Log
    </button>
  </div>
</div>
)}

{processState === 'completed' && showSuccessMessage && (
<div className="movie-page__success-message">
  <h2 className="movie-page__success-title">Processamento Concluído</h2>
  <p>Todas as séries foram processadas com sucesso!</p>
  <div className="movie-page__final-results">
    <h3>Resultados Finais:</h3>
    <div className="movie-page__stats-grid">
      <div className="movie-page__stats-card">
        <FaTv />
        <h4>Total Processado</h4>
        <p>{progress.processedItems}</p>
      </div>
      <div className="movie-page__stats-card">
        <FaCheckCircle />
        <h4>Séries Adicionadas</h4>
        <p>{progress.addedItems}</p>
      </div>
      <div className="movie-page__stats-card">
        <FaExclamationTriangle />
        <h4>Séries Duplicadas</h4>
        <p>{progress.duplicateSeries}</p>
      </div>
    </div>
    <div className="movie-page__stats-grid">
      <div className="movie-page__stats-card">
        <h4>Total de Episódios</h4>
        <p>{progress.totalEpisodes}</p>
      </div>
      <div className="movie-page__stats-card">
        <h4>Episódios Adicionados</h4>
        <p>{progress.addedEpisodes}</p>
      </div>
      <div className="movie-page__stats-card">
        <h4>Episódios Duplicados</h4>
        <p>{progress.duplicateEpisodes}</p>
      </div>
    </div>
  </div>
  <div className="movie-page__log-container">
    <h3 className="movie-page__subtitle">Log Final de Processamento:</h3>
    <ul className="movie-page__log-list">
      {logMessages.map((message, index) => (
        <li key={index}>{message}</li>
      ))}
    </ul>
  </div>
  <div className="movie-page__button-group">
    <button onClick={copyLogToClipboard} className="movie-page__btn movie-page__btn--secondary">
      <FaCopy /> Copiar Log Completo
    </button>
    <button onClick={handleCancelProcess} className="movie-page__btn movie-page__btn--primary">
      <FaStop /> Finalizar e Limpar
    </button>
  </div>
  <p className="movie-page__note">
    Nota: Você pode copiar o log completo ou finalizar o processo. 
    Finalizar irá limpar os dados de processamento e permitir que você inicie um novo processo.
  </p>
</div>
)}

{/* Footer or additional information */}
<footer className="movie-page__footer">
  <p>© 2024 Sync Séries. Todos os direitos reservados.</p>
</footer>
</div>
);
};

export default SeriesPage;