import React, { useState, useEffect, useRef } from 'react';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';
import io from 'socket.io-client';
import { FaPlay, FaPause, FaStop, FaCopy, FaFilm, FaCheckCircle, FaExclamationTriangle, FaCheck, FaTimes } from 'react-icons/fa';

const BACKEND_URL = 'http://sync.zpainel.site:5000';

const MoviePage = () => {
  const [url, setUrl] = useState('');
  const [categories, setCategories] = useState([]);
  const [totalMovies, setTotalMovies] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState('');
  const [processState, setProcessState] = useState('idle');
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [progress, setProgress] = useState({
    processedItems: 0,
    addedItems: 0,
    duplicateMovies: 0,
    timeRemaining: 0,
    newMovies: [],
    totalItems: 0
  });
  const [bouquets, setBouquets] = useState([]);
  const [selectedBouquet, setSelectedBouquet] = useState('');
  const [logMessages, setLogMessages] = useState([]);
  const [allSelected, setAllSelected] = useState(false);
  const navigate = useNavigate();
  const socketRef = useRef(null);

  useEffect(() => {
    const token = localStorage.getItem('token');
    if (token) {
      setIsAuthenticated(true);
      axios.defaults.headers.common['Authorization'] = `Bearer ${token}`;
      fetchBouquets();
      checkProcessStatus();
    } else {
      navigate('/login');
    }
  }, [navigate]);

  useEffect(() => {
    if (isAuthenticated) {
      socketRef.current = io(BACKEND_URL, {
        query: { token: localStorage.getItem('token') }
      });

      socketRef.current.on('connect', () => {
        console.log('Socket connected');
      });

      socketRef.current.on('connect_error', (error) => {
        console.error('Socket connection error:', error);
      });

      socketRef.current.on('processUpdate', handleProcessUpdate);
      socketRef.current.on('processComplete', handleProcessComplete);
      socketRef.current.on('processError', handleProcessError);

      return () => {
        if (socketRef.current) {
          socketRef.current.disconnect();
        }
      };
    }
  }, [isAuthenticated]);

  const fetchBouquets = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/movies/bouquets`);
      if (response.data && response.data.length > 0) {
        setBouquets(response.data);
        setSelectedBouquet(response.data[0].id);
      } else {
        setError('Nenhum bouquet encontrado. Por favor, crie um bouquet primeiro.');
      }
    } catch (err) {
      console.error('Error fetching bouquets:', err);
      handleError(err);
    }
  };

  const checkProcessStatus = async () => {
    try {
      const response = await axios.get(`${BACKEND_URL}/api/movies/process-status`);
      if (response.data.isRunning || response.data.isCompleted) {
        setProcessState(response.data.isPaused ? 'paused' : response.data.isCompleted ? 'completed' : 'running');
        setProgress(response.data.progress);
        if (response.data.isCompleted) {
          setLogMessages(response.data.progress.log || []);
        }
      }
    } catch (err) {
      console.error('Error checking process status:', err);
    }
  };

  const handleAnalyze = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError('');
    setCategories([]);
    setTotalMovies(0);
    setProcessState('idle');

    try {
      const response = await axios.post(`${BACKEND_URL}/api/movies/analyze`, { url });
      setCategories(response.data.categories.map(category => ({
        ...category,
        selected: false,
        newName: category.category_name
      })));
      setTotalMovies(response.data.totalMovies);
    } catch (err) {
      handleError(err);
    } finally {
      setIsLoading(false);
    }
  };

  const handleCategorySelect = (categoryId) => {
    setCategories(categories.map(category =>
      category.category_id === categoryId ? { ...category, selected: !category.selected } : category
    ));
    updateAllSelectedState();
  };

  const handleCategoryRename = (categoryId, newName) => {
    setCategories(categories.map(category =>
      category.category_id === categoryId ? { ...category, newName } : category
    ));
  };

  const handleSelectAllCategories = () => {
    setCategories(categories.map(category => ({ ...category, selected: true })));
    setAllSelected(true);
  };

  const handleDeselectAllCategories = () => {
    setCategories(categories.map(category => ({ ...category, selected: false })));
    setAllSelected(false);
  };

  const updateAllSelectedState = () => {
    setAllSelected(categories.every(category => category.selected));
  };

  const handleStartProcess = async () => {
    setIsLoading(true);
    setError('');
    setProcessState('running');
    setProgress({
      processedItems: 0,
      addedItems: 0,
      duplicateMovies: 0,
      timeRemaining: 0,
      newMovies: [],
      totalItems: 0
    });
    setLogMessages([]);

    const selectedCategories = categories
      .filter(category => category.selected)
      .map(({ category_id, newName }) => ({
        id: category_id,
        newName,
        selected: true
      }));

    console.log('Selected categories:', selectedCategories);

    if (selectedCategories.length === 0) {
      setError('Por favor, selecione pelo menos uma categoria.');
      setIsLoading(false);
      setProcessState('idle');
      return;
    }

    try {
      await axios.post(`${BACKEND_URL}/api/movies/process`, {
        url,
        bouquetId: selectedBouquet,
        selectedCategories
      });
    } catch (err) {
      handleError(err);
      setProcessState('idle');
    } finally {
      setIsLoading(false);
    }
  };

  const handlePauseProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/movies/pause`);
      setProcessState('paused');
    } catch (err) {
      handleError(err);
    }
  };

  const handleResumeProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/movies/resume`);
      setProcessState('running');
    } catch (err) {
      handleError(err);
    }
  };

  const handleCancelProcess = async () => {
    try {
      await axios.post(`${BACKEND_URL}/api/movies/cancel`);
      setProcessState('idle');
      setProgress({
        processedItems: 0,
        addedItems: 0,
        duplicateMovies: 0,
        timeRemaining: 0,
        newMovies: [],
        totalItems: 0
      });
      setLogMessages([]);
    } catch (err) {
      handleError(err);
    }
  };

  const handleProcessUpdate = (data) => {
    setProgress(prevProgress => ({
      ...prevProgress,
      ...data
    }));
    if (data.log && data.log.length > 0) {
      setLogMessages(prevLogs => [...prevLogs, ...data.log]);
    }
  };

  const handleProcessComplete = (data) => {
    setProcessState('completed');
    setProgress(prevProgress => ({
      ...prevProgress,
      ...data
    }));
    setLogMessages(prevLogs => [...prevLogs, ...(data.log || [])]);
  };

  const handleProcessError = (error) => {
    setError(error.message);
    setProcessState('idle');
  };

  const handleError = (err) => {
    if (err.response && err.response.status === 401) {
      setError('Sessão expirada. Por favor, faça login novamente.');
      setIsAuthenticated(false);
      localStorage.removeItem('token');
      navigate('/login');
    } else {
      setError(err.response?.data?.message || err.message || 'Ocorreu um erro');
    }
  };

  const copyLogToClipboard = () => {
    const logText = logMessages.join('\n');
    navigator.clipboard.writeText(logText)
      .then(() => alert('Log copiado para a área de transferência!'))
      .catch(err => {
        console.error('Falha ao copiar log:', err);
        alert('Falha ao copiar o log. Por favor, tente copiar manualmente.');
      });
  };

  if (!isAuthenticated) {
    return <p>Por favor, faça login para acessar esta página.</p>;
  }

return (
  <div className="movie-page">
    <h1 className="movie-page__title">Sync Filmes</h1>
    <p>Sincronize filmes de forma rápida e segura para seu servidor.</p>
    {processState === 'idle' && (
      <div className="movie-page__card">
        <form onSubmit={handleAnalyze} className="movie-page__form">
          <div className="movie-page__form-group">
            <label htmlFor="xtream-url">Insira a URL da Fonte:</label>
            <input
              type="text"
              id="xtream-url"
              value={url}
              onChange={(e) => setUrl(e.target.value)}
              required
              placeholder="http://exemplo.com:porta/..."
            />
          </div>
          <button 
            type="submit" 
            disabled={isLoading}
            className="movie-page__btn movie-page__btn--primary"
          >
            {isLoading ? 'Processando...' : 'Analisar'}
          </button>
        </form>
      </div>
    )}

    {error && <p className="movie-page__error">{error}</p>}

    {categories.length > 0 && processState === 'idle' && (
      <div className="movie-page__card">
        <h2 className="movie-page__card-title">Categorias</h2>
        <div className="movie-page__category-controls">
          <button 
            onClick={handleSelectAllCategories}
            className="movie-page__btn movie-page__btn--secondary"
          >
            <FaCheck /> Selecionar Todas
          </button>
          <button 
            onClick={handleDeselectAllCategories}
            className="movie-page__btn movie-page__btn--secondary"
          >
            <FaTimes /> Desmarcar Todas
          </button>
        </div>
        <div className="movie-page__categories-list">
          {categories.map(category => (
            <div key={category.category_id} className="movie-page__category-item">
              <input
                type="checkbox"
                id={`category-${category.category_id}`}
                checked={category.selected}
                onChange={() => handleCategorySelect(category.category_id)}
              />
              <input
                type="text"
                value={category.newName}
                onChange={(e) => handleCategoryRename(category.category_id, e.target.value)}
              />
              <span>{category.movieCount} filmes</span>
            </div>
          ))}
        </div>
        
        <div className="movie-page__stats-grid">
          <div className="movie-page__stats-card">
            <h3>Total de categorias</h3>
            <p>{categories.length}</p>
          </div>
          <div className="movie-page__stats-card">
            <h3>Total de filmes</h3>
            <p>{totalMovies}</p>
          </div>
        </div>
    
        {bouquets.length > 0 ? (
          <div className="movie-page__form-group">
            <label htmlFor="bouquet-select">Selecione o Bouquet:</label>
            <select
              id="bouquet-select"
              value={selectedBouquet}
              onChange={(e) => setSelectedBouquet(e.target.value)}
              required
            >
              {bouquets.map((bouquet) => (
                <option key={bouquet.id} value={bouquet.id}>
                  {bouquet.bouquet_name}
                </option>
              ))}
            </select>
          </div>
        ) : (
          <p className="movie-page__error">Nenhum bouquet disponível. Por favor, crie um bouquet primeiro.</p>
        )}
        
        <button 
          onClick={handleStartProcess} 
          disabled={isLoading || processState !== 'idle' || !selectedBouquet || categories.filter(c => c.selected).length === 0}
          className="movie-page__btn movie-page__btn--success"
        >
          <FaPlay /> Iniciar Processamento
        </button>
      </div>
    )}

    {(processState === 'running' || processState === 'paused') && (
      <div className="movie-page__card">
        <h2 className="movie-page__card-title">Estatísticas</h2>
        <div className="movie-page__progress-bar">
          <div 
            className="movie-page__progress"
            style={{width: `${(progress.processedItems / progress.totalItems) * 100}%`}}
          ></div>
        </div>
        <div className="movie-page__stats-grid">
          <div className="movie-page__stats-card">
            <FaFilm />
            <h3>Filmes Processados</h3>
            <p>{progress.processedItems} / {progress.totalItems}</p>
          </div>
          <div className="movie-page__stats-card">
            <FaCheckCircle />
            <h3>Filmes Adicionados</h3>
            <p>{progress.addedItems}</p>
          </div>
          <div className="movie-page__stats-card">
            <FaExclamationTriangle />
            <h3>Filmes Duplicados</h3>
            <p>{progress.duplicateMovies}</p>
          </div>
          <div className="movie-page__stats-card">
            <h3>Tempo Restante Estimado</h3>
            <p>{Math.floor(progress.timeRemaining / 60)} minutos</p>
          </div>
        </div>
        <div className="movie-page__log-container">
          <h3 className="movie-page__subtitle">Log de Processamento:</h3>
          <ul className="movie-page__log-list">
            {logMessages.map((message, index) => (
              <li key={index}>{message}</li>
            ))}
          </ul>
        </div>
        <div className="movie-page__button-group">
          {processState === 'running' && (
            <button onClick={handlePauseProcess} className="movie-page__btn movie-page__btn--warning">
              <FaPause /> Pausar
            </button>
          )}
          {processState === 'paused' && (
            <button onClick={handleResumeProcess} className="movie-page__btn movie-page__btn--success">
              <FaPlay /> Continuar
            </button>
          )}
          <button onClick={handleCancelProcess} className="movie-page__btn movie-page__btn--danger">
            <FaStop /> Cancelar
          </button>
          <button onClick={copyLogToClipboard} className="movie-page__btn movie-page__btn--secondary">
            <FaCopy /> Copiar Log
          </button>
        </div>
      </div>
    )}

    {processState === 'completed' && (
      <div className="movie-page__success-message">
        <h2 className="movie-page__success-title">Processamento Concluído</h2>
        <p>Todos os filmes foram processados com sucesso!</p>
        <div className="movie-page__final-results">
          <h3>Resultados Finais:</h3>
          <div className="movie-page__stats-grid">
            <div className="movie-page__stats-card">
              <FaFilm />
              <h4>Total Processado</h4>
              <p>{progress.processedItems}</p>
            </div>
            <div className="movie-page__stats-card">
              <FaCheckCircle />
              <h4>Filmes Adicionados</h4>
              <p>{progress.addedItems}</p>
            </div>
            <div className="movie-page__stats-card">
              <FaExclamationTriangle />
              <h4>Filmes Duplicados</h4>
              <p>{progress.duplicateMovies}</p>
            </div>
          </div>
        </div>
        <div className="movie-page__log-container">
          <h3 className="movie-page__subtitle">Log Final de Processamento:</h3>
          <ul className="movie-page__log-list">
            {logMessages.map((message, index) => (
              <li key={index}>{message}</li>
            ))}
          </ul>
        </div>
        <div className="movie-page__button-group">
          <button onClick={copyLogToClipboard} className="movie-page__btn movie-page__btn--secondary">
            <FaCopy /> Copiar Log Completo
          </button>
          <button onClick={handleCancelProcess} className="movie-page__btn movie-page__btn--primary">
            <FaStop /> Finalizar e Limpar
          </button>
        </div>
        <p className="movie-page__note">
          Nota: Você pode copiar o log completo ou finalizar o processo. 
          Finalizar irá limpar os dados de processamento e permitir que você inicie um novo processo.
        </p>
      </div>
    )}

    {/* Footer or additional information */}
    <footer className="movie-page__footer">
      <p>© 2024 Sync Filmes. Todos os direitos reservados.</p>
    </footer>
  </div>
);
};

export default MoviePage;