import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { FaUser, FaLock } from 'react-icons/fa';
import { useAuth } from './AuthContext';

const Notification = ({ message, type }) => (
  <div className={`notification ${type}`}>
    {message}
  </div>
);

const LoginPage = () => {
  const [credentials, setCredentials] = useState({ username: '', password: '' });
  const [notification, setNotification] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const navigate = useNavigate();
  const { login } = useAuth();

  const handleChange = (e) => {
    const { name, value } = e.target;
    setCredentials(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setNotification(null);
    setIsLoading(true);
    try {
      const success = await login(credentials.username, credentials.password);
      if (success) {
        setNotification({ message: 'Login realizado com sucesso!', type: 'success' });
        setTimeout(() => navigate('/'), 1500);
      } else {
        setNotification({ message: 'Credenciais inválidas. Por favor, tente novamente.', type: 'error' });
      }
    } catch (err) {
      console.error('Login error:', err);
      setNotification({ 
        message: err.response?.data?.message || 'Erro ao fazer login. Por favor, tente novamente.', 
        type: 'error' 
      });
    } finally {
      setIsLoading(false);
    }
  };
  
  return (
    <div className="login-page">
      <div className="login-container">
        {notification && <Notification message={notification.message} type={notification.type} />}
        <form className="login-form card" onSubmit={handleSubmit}>
          <h2 className="card-title">Login</h2>
          <div className="form-group">
            <label htmlFor="username">
              <FaUser /> Usuário
            </label>
            <input
              id="username"
              name="username"
              type="text"
              value={credentials.username}
              onChange={handleChange}
              required
              autoComplete="username"
            />
          </div>
          <div className="form-group">
            <label htmlFor="password">
              <FaLock /> Senha
            </label>
            <input
              id="password"
              name="password"
              type="password"
              value={credentials.password}
              onChange={handleChange}
              required
              autoComplete="current-password"
            />
          </div>
          <button 
            type="submit" 
            className="btn btn-primary"
            disabled={isLoading}
          >
            {isLoading ? 'Entrando...' : 'Entrar'}
          </button>
        </form>
      </div>
    </div>
  );
};

export default LoginPage;