import React, { useState, useEffect } from 'react';
import axios from '../utils/axiosConfig';
import { FaSpinner, FaCheck, FaTimes, FaFilm, FaTv, FaCalendarAlt, FaClock, FaServer, FaDatabase, FaUserCog, FaBook, FaChevronRight, FaCopy, FaSave } from 'react-icons/fa';

const HomePage = () => {
  const [syncStatus, setSyncStatus] = useState('unknown');
  const [dbStatus, setDbStatus] = useState({ status: 'unknown', responseTime: null });
  const [userInfo, setUserInfo] = useState({ subscriptionType: '', daysLeft: 0 });
  const [tutorials, setTutorials] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [dbCredentials, setDbCredentials] = useState({ host: '', user: '', password: '' });
  const [isDbConfigured, setIsDbConfigured] = useState(true);

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        setError(null);

        const [syncResponse, dbResponse, userInfoResponse, tutorialsResponse] = await Promise.all([
          axios.get('/api/home/sync/status'),
          axios.get('/api/home/db/status'),
          axios.get('/api/home/user/info'),
          axios.get('/api/home/tutorials')
        ]);

        setSyncStatus(syncResponse.data.status);
        setDbStatus({
          status: dbResponse.data.status,
          responseTime: dbResponse.data.responseTime
        });
        setUserInfo(userInfoResponse.data);
        setTutorials(tutorialsResponse.data);
        setIsDbConfigured(dbResponse.data.isConfigured);
      } catch (error) {
        console.error('Error fetching data:', error);
        setError('Ocorreu um erro ao carregar os dados. Por favor, tente novamente mais tarde.');
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const handleDbCredentialsChange = (e) => {
    const { name, value } = e.target;
    setDbCredentials(prev => ({ ...prev, [name]: value }));
  };

  const handleSaveDbCredentials = async () => {
    try {
      setLoading(true);
      const response = await axios.post('/api/home/save-db-credentials', dbCredentials);
      if (response.data.success) {
        setIsDbConfigured(true);
        setDbStatus({ status: 'online', responseTime: response.data.responseTime });
      } else {
        setError('Não foi possível conectar ao banco de dados. Verifique suas credenciais.');
      }
    } catch (error) {
      console.error('Error saving DB credentials:', error);
      setError('Ocorreu um erro ao salvar as credenciais. Por favor, tente novamente.');
    } finally {
      setLoading(false);
    }
  };

    const copyToClipboard = (text) => {
    if (navigator.clipboard && navigator.clipboard.writeText) {
      // Use Clipboard API if available
      navigator.clipboard.writeText(text)
        .then(() => alert('Texto copiado para a área de transferência!'))
        .catch(err => console.error('Erro ao copiar texto: ', err));
    } else {
      // Fallback method using a temporary textarea element
      const textArea = document.createElement("textarea");
      textArea.value = text;
      document.body.appendChild(textArea);
      textArea.select();
      try {
        document.execCommand('copy');
        alert('Texto copiado para a área de transferência!');
      } catch (err) {
        console.error('Erro ao copiar texto: ', err);
      }
      document.body.removeChild(textArea);
    }
  };
  const renderTutorials = () => {
    return tutorials.map((tutorial, index) => (
      <div key={index} className="tutorial-card">
        <h3>{tutorial.title}</h3>
        {tutorial.content.split(/<pre class="copyable-text">([\s\S]*?)<\/pre>/g).map((part, i) => {
          if (i % 2 === 0) {
            return <div key={i} dangerouslySetInnerHTML={{ __html: part }} />;
          } else {
            return (
              <pre key={i} className="copyable-text">
                {part}
                <button className="copy-button" onClick={() => copyToClipboard(part)}>
                  <FaCopy />
                </button>
              </pre>
            );
          }
        })}
      </div>
    ));
  };
  if (loading) {
    return (
      <div className="home-page">
        <div className="card loading-card">
          <FaSpinner className="icon-spin" />
          <p>Carregando informações...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="home-page">
        <div className="card error-card">
          <FaTimes className="error-icon" />
          <h2>Erro</h2>
          <p>{error}</p>
        </div>
      </div>
    );
  }

  return (
    <div className="home-page">

      <div className="dashboard-grid">
        <div className="dashboard-card">
          <div className="card-icon">
            <FaServer />
          </div>
          <div className="card-content">
            <h3>Sincronizador</h3>
            <div className={`status ${syncStatus === 'online' ? 'status-online' : 'status-offline'}`}>
              {syncStatus === 'online' ? <FaCheck /> : <FaTimes />}
              <span>{syncStatus === 'online' ? 'Online' : 'Offline'}</span>
            </div>
          </div>
        </div>
        
        <div className="dashboard-card">
          <div className="card-icon">
            <FaDatabase />
          </div>
          <div className="card-content">
            <h3>Banco de Dados</h3>
            <div className={`status ${dbStatus.status === 'online' ? 'status-online' : 'status-offline'}`}>
              {dbStatus.status === 'online' ? <FaCheck /> : <FaTimes />}
              <span>{dbStatus.status === 'online' ? 'Online' : 'Offline'}</span>
            </div>
            {dbStatus.status === 'online' && (
              <p className="response-time">({dbStatus.responseTime}ms)</p>
            )}
          </div>
        </div>
        
        <div className="dashboard-card">
          <div className="card-icon">
            <FaUserCog />
          </div>
          <div className="card-content">
            <h3>Assinatura</h3>
            <p className="subscription-type">
              {userInfo.subscriptionType === 'Só Filmes' && <FaFilm />}
              {userInfo.subscriptionType === 'Só Séries' && <FaTv />}
              {userInfo.subscriptionType === 'Filmes e Séries' && <><FaFilm /> <FaTv /></>}
              <span>{userInfo.subscriptionType}</span>
            </p>
            <p className="subscription-days">
              <FaCalendarAlt />
              <span>{userInfo.daysLeft} dias restantes</span>
            </p>
          </div>
        </div>
      </div>
      
      {!isDbConfigured && (
        <div className="db-config-section">
          <h2 className="section-title"><FaDatabase /> Configuração do Banco de Dados</h2>
          <div className="db-config-form">
            <input
              type="text"
              name="host"
              placeholder="Host do Banco de Dados"
              value={dbCredentials.host}
              onChange={handleDbCredentialsChange}
            />
            <input
              type="text"
              name="user"
              placeholder="Usuário do Banco de Dados"
              value={dbCredentials.user}
              onChange={handleDbCredentialsChange}
            />
            <input
              type="password"
              name="password"
              placeholder="Senha do Banco de Dados"
              value={dbCredentials.password}
              onChange={handleDbCredentialsChange}
            />
            <button className="btn btn-primary" onClick={handleSaveDbCredentials}>
              <FaSave /> Salvar e Testar Conexão
            </button>
          </div>
        </div>
      )}
      
      <div className="tutorials-section">
        <h2 className="section-title"><FaBook /> Tutoriais</h2>
    {renderTutorials()}
      </div>
    </div>
  );
};

export default HomePage;