import React, { createContext, useContext, useState, useEffect } from 'react';
import axios from 'axios';

const AuthContext = createContext();

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const checkAuth = () => {
      const token = localStorage.getItem('token');
      const username = localStorage.getItem('username');
      const role = localStorage.getItem('role');
      const expiresIn = localStorage.getItem('expiresIn');

      if (token && username && role && expiresIn) {
        const expirationDate = new Date(expiresIn);
        if (expirationDate > new Date()) {
          setUser({ username, role });
          setAuthToken(token);
        } else {
          logout();
        }
      }
      setLoading(false);
    };

    checkAuth();
  }, []);

  const setAuthToken = (token) => {
    if (token) {
      axios.defaults.headers.common['Authorization'] = `Bearer ${token}`;
    } else {
      delete axios.defaults.headers.common['Authorization'];
    }
  };

  const login = async (username, password) => {
    try {
      const response = await axios.post('http://sync.zpainel.site:5000/api/auth/login', { username, password });
      const { token, role, expiresIn, username: loggedInUsername } = response.data;
      
      localStorage.setItem('token', token);
      localStorage.setItem('role', role);
      localStorage.setItem('expiresIn', expiresIn);
      localStorage.setItem('username', loggedInUsername);
      
      setAuthToken(token);
      setUser({ username: loggedInUsername, role });
      return true;
    } catch (error) {
      console.error('Login error:', error);
      return false;
    }
  };

  const logout = () => {
    localStorage.removeItem('token');
    localStorage.removeItem('role');
    localStorage.removeItem('expiresIn');
    localStorage.removeItem('username');
    setAuthToken(null);
    setUser(null);
  };

  return (
    <AuthContext.Provider value={{ user, login, logout, loading }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => useContext(AuthContext);

// Configuração global do Axios
axios.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response && error.response.status === 401) {
      // Logout automático se receber um 401
      localStorage.clear();
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);